const hre = require("hardhat");
const fs = require('fs');

async function main() {
    console.log('\n' + '='.repeat(80));
    console.log('SMART CONTRACT VERIFICATION - AUTO FIX');
    console.log('='.repeat(80));

    // Get deployed addresses
    const deployed = JSON.parse(fs.readFileSync('deployed-addresses.json', 'utf8'));
    const tokenAddress = deployed.tokenAddress;
    const stakingAddress = deployed.stakingAddress;

    console.log('\nToken Address:', tokenAddress);
    console.log('Staking Address:', stakingAddress);

    // Wait for BSCScan to index
    console.log('\n⏳ Waiting 15 seconds for BSCScan indexing...');
    await new Promise(resolve => setTimeout(resolve, 15000));

    // Verify Token Contract
    console.log('\n' + '='.repeat(80));
    console.log('1️⃣  VERIFYING TOKEN CONTRACT');
    console.log('='.repeat(80));

    try {
        console.log('Starting verification...');
        await hre.run("verify:verify", {
            address: tokenAddress,
            constructorArguments: []
        });
        console.log('✅ Token verified successfully!');
    } catch (error) {
        if (error.message.toLowerCase().includes('already verified')) {
            console.log('✅ Token already verified!');
        } else {
            console.log('⚠️  Token verification issue:', error.message);
            if (error.message.includes('does not have bytecode')) {
                console.log('   → Contract might still be indexing, try again in 1 minute');
            } else if (error.message.includes('compilation failed')) {
                console.log('   → Source code mismatch detected');
            }
        }
    }

    // Wait between verifications
    console.log('\n⏳ Waiting 5 seconds...');
    await new Promise(resolve => setTimeout(resolve, 5000));

    // Verify Staking Contract  
    console.log('\n' + '='.repeat(80));
    console.log('2️⃣  VERIFYING STAKING CONTRACT');
    console.log('='.repeat(80));

    try {
        console.log('Starting verification...');
        console.log('Constructor Arg: [', tokenAddress, ']');

        await hre.run("verify:verify", {
            address: stakingAddress,
            constructorArguments: [tokenAddress]
        });
        console.log('✅ Staking verified successfully!');
    } catch (error) {
        if (error.message.toLowerCase().includes('already verified')) {
            console.log('✅ Staking already verified!');
        } else {
            console.log('⚠️  Staking verification issue:', error.message);
            if (error.message.includes('does not have bytecode')) {
                console.log('   → Contract might still be indexing, try again in 1 minute');
            } else if (error.message.includes('compilation failed')) {
                console.log('   → Source code mismatch detected');
            }
        }
    }

    console.log('\n' + '='.repeat(80));
    console.log('VERIFICATION ATTEMPT COMPLETE');
    console.log('='.repeat(80));

    console.log('\n📊 Check verification status:');
    console.log(`Token:   https://testnet.bscscan.com/address/${tokenAddress}#code`);
    console.log(`Staking: https://testnet.bscscan.com/address/${stakingAddress}#code`);

    console.log('\n💡 If verification failed:');
    console.log('1. Check if BSCScan API is working');
    console.log('2. Wait 1 minute and run: npm run verify');
    console.log('3. Or use manual verification guide');

    console.log('\n' + '='.repeat(80) + '\n');
}

main()
    .then(() => process.exit(0))
    .catch((error) => {
        console.error('\n❌ Critical Error:', error);
        process.exit(1);
    });
