const { ethers } = require("hardhat");

async function main() {
    const [owner] = await ethers.getSigners();

    // Get contract addresses from .env
    const TOKEN_ADDRESS = process.env.KZD_TOKEN_ADDRESS;
    const STAKING_ADDRESS = process.env.STAKING_CONTRACT_ADDRESS;

    if (!TOKEN_ADDRESS || !STAKING_ADDRESS) {
        console.error("❌ Contract addresses not found in .env");
        return;
    }

    console.log("========================================");
    console.log("FUND CONTRACT");
    console.log("========================================");
    console.log("Owner:", owner.address);
    console.log("Token:", TOKEN_ADDRESS);
    console.log("Staking:", STAKING_ADDRESS);
    console.log("");

    // Attach to contracts
    const Token = await ethers.getContractAt("KUZADESIGNToken", TOKEN_ADDRESS);
    const Staking = await ethers.getContractAt("KZDStaking", STAKING_ADDRESS);

    // Amount to fund (default: 10,000 KZD)
    // Change this value as needed
    const FUND_AMOUNT = process.env.FUND_AMOUNT || "10000";
    const amount = ethers.parseEther(FUND_AMOUNT);

    console.log("💰 Amount to fund:", ethers.formatEther(amount), "KZD");
    console.log("");

    // Step 1: Check owner balance
    console.log("📊 Checking owner balance...");
    const ownerBalance = await Token.balanceOf(owner.address);
    console.log("   Owner balance:", ethers.formatEther(ownerBalance), "KZD");

    if (ownerBalance < amount) {
        console.error("❌ Insufficient balance!");
        console.error(`   Need: ${ethers.formatEther(amount)} KZD`);
        console.error(`   Have: ${ethers.formatEther(ownerBalance)} KZD`);
        return;
    }

    // Step 2: Check current contract balance
    console.log("📊 Checking contract balance...");
    const currentBalance = await Staking.getContractBalance();
    console.log("   Current balance:", ethers.formatEther(currentBalance), "KZD");
    console.log("");

    // Step 3: Approve
    console.log("🔓 Approving token...");
    const approveTx = await Token.approve(STAKING_ADDRESS, amount);
    console.log("   TX:", approveTx.hash);
    await approveTx.wait();
    console.log("✅ Approved!");
    console.log("");

    // Step 4: Fund contract
    console.log("💸 Funding contract...");
    const fundTx = await Staking.fundContract(amount);
    console.log("   TX:", fundTx.hash);
    const receipt = await fundTx.wait();
    console.log("✅ Contract funded!");
    console.log("");

    // Step 5: Check new balance
    console.log("📊 New contract balance:");
    const newBalance = await Staking.getContractBalance();
    console.log("   Balance:", ethers.formatEther(newBalance), "KZD");
    console.log("   Increase:", ethers.formatEther(newBalance - currentBalance), "KZD");
    console.log("");

    console.log("========================================");
    console.log("✅ FUNDING SUCCESSFUL!");
    console.log("========================================");
    console.log("Transaction:", receipt.hash);
    console.log("Block:", receipt.blockNumber);
}

main()
    .then(() => process.exit(0))
    .catch((error) => {
        console.error(error);
        process.exit(1);
    });
