require('dotenv').config();
const mongoose = require('mongoose');
const connectDB = require('../lib/mongodb');
const Stake = require('../models/Stake');
const Balance = require('../models/Balance');

/**
 * Daily Rewards Calculation Script
 * Runs every day to calculate and distribute daily rewards
 */
async function calculateDailyRewards() {
    console.log('🕐 Starting daily rewards calculation...');
    console.log('📅 Date:', new Date().toISOString());

    try {
        // Connect to database
        await connectDB();

        // Get all active stakes
        const activeStakes = await Stake.find({ status: 'active' });
        console.log(`📊 Found ${activeStakes.length} active stakes`);

        let totalRewardsDistributed = 0;
        let stakesCompleted = 0;

        for (const stake of activeStakes) {
            try {
                // Calculate daily reward
                // dailyRate is in basis points (1.5% = 150)
                const dailyReward = (stake.amount * stake.dailyRate) / 10000;

                // Update user balance
                let balance = await Balance.findOne({ userWallet: stake.userWallet });

                if (!balance) {
                    balance = new Balance({ userWallet: stake.userWallet });
                }

                balance.dailyRewards += dailyReward;
                await balance.save();

                // Update stake total claimed
                stake.totalClaimed += dailyReward;

                // Check if stake has ended
                if (new Date() >= new Date(stake.endDate)) {
                    stake.status = 'completed';
                    stakesCompleted++;
                    console.log(`✅ Stake completed for ${stake.userWallet} - Package: ${stake.packageName}`);
                }

                await stake.save();

                totalRewardsDistributed += dailyReward;

                console.log(`💰 Reward: ${dailyReward} KZD → ${stake.userWallet} (${stake.packageName})`);

            } catch (error) {
                console.error(`❌ Error processing stake ${stake._id}:`, error);
            }
        }

        console.log('\n' + '='.repeat(60));
        console.log('📈 DAILY REWARDS SUMMARY');
        console.log('='.repeat(60));
        console.log(`✅ Total rewards distributed: ${totalRewardsDistributed.toFixed(2)} KZD`);
        console.log(`📦 Stakes processed: ${activeStakes.length}`);
        console.log(`🏁 Stakes completed: ${stakesCompleted}`);
        console.log(`🔄 Stakes still active: ${activeStakes.length - stakesCompleted}`);
        console.log('='.repeat(60));

    } catch (error) {
        console.error('❌ Error in daily rewards calculation:', error);
        throw error;
    } finally {
        // Close MongoDB connection
        await mongoose.connection.close();
        console.log('🔌 Database connection closed');
    }
}

// Run the script
calculateDailyRewards()
    .then(() => {
        console.log('✨ Daily rewards calculation completed successfully');
        process.exit(0);
    })
    .catch((error) => {
        console.error('💥 Daily rewards calculation failed:', error);
        process.exit(1);
    });
