import { useState, useEffect } from 'react';
import Head from 'next/head';
import BottomNav from '../components/BottomNav';

export default function Withdraw() {
    const [account, setAccount] = useState(null);
    const [balance, setBalance] = useState({ dailyRewards: 0, referralBonus: 0, managerBonus: 0 });
    const [activeTab, setActiveTab] = useState('daily');
    const [amount, setAmount] = useState('');
    const [isWithdrawing, setIsWithdrawing] = useState(false);

    useEffect(() => {
        checkWallet();
    }, []);

    useEffect(() => {
        if (account) {
            fetchBalance();
        }
    }, [account]);

    const checkWallet = async () => {
        if (typeof window.ethereum !== 'undefined') {
            const accounts = await window.ethereum.request({ method: 'eth_accounts' });
            if (accounts.length > 0) {
                setAccount(accounts[0]);
            }
        }
    };

    const fetchBalance = async () => {
        try {
            const response = await fetch(`/api/balance/get?userWallet=${account}`);
            const data = await response.json();
            if (data.success) {
                setBalance(data.balance);
            }
        } catch (error) {
            console.error('Error fetching balance:', error);
        }
    };

    const getAvailableBalance = () => {
        if (activeTab === 'daily') return balance.dailyRewards;
        if (activeTab === 'referral') return balance.referralBonus;
        return balance.managerBonus;
    };

    const getWithdrawType = () => {
        if (activeTab === 'daily') return 'withdraw_daily';
        if (activeTab === 'referral') return 'withdraw_referral';
        return 'withdraw_manager';
    };

    const handleWithdraw = async () => {
        const withdrawAmount = parseFloat(amount);

        if (!withdrawAmount || withdrawAmount < 10) {
            alert('Minimum withdrawal amount is 10 KZD');
            return;
        }

        if (withdrawAmount > getAvailableBalance()) {
            alert('Insufficient balance');
            return;
        }

        const fee = withdrawAmount * 0.05;
        const received = withdrawAmount - fee;

        if (!confirm(`Withdraw ${withdrawAmount} KZD?\n\nFee (5%): ${fee.toFixed(2)} KZD\nYou will receive: ${received.toFixed(2)} KZD`)) {
            return;
        }

        setIsWithdrawing(true);

        try {
            const { ethers } = await import('ethers');
            const provider = new ethers.BrowserProvider(window.ethereum);
            const signer = await provider.getSigner();

            const STAKING_ADDRESS = process.env.NEXT_PUBLIC_STAKING_CONTRACT_ADDRESS || '0x0000000000000000000000000000000000000000';

            if (STAKING_ADDRESS === '0x0000000000000000000000000000000000000000') {
                alert('Contract address not configured');
                setIsWithdrawing(false);
                return;
            }

            const { STAKING_CONTRACT_ABI } = await import('../lib/web3Config');
            const stakingContract = new ethers.Contract(STAKING_ADDRESS, STAKING_CONTRACT_ABI, signer);

            // Call withdraw function on contract
            const amountWei = ethers.parseEther(withdrawAmount.toString());
            const withdrawTx = await stakingContract.withdraw(amountWei);
            const receipt = await withdrawTx.wait();

            // Record withdrawal in database
            const response = await fetch('/api/withdraw/request', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({
                    userWallet: account,
                    amount: withdrawAmount,
                    type: getWithdrawType(),
                    txHash: receipt.hash,
                }),
            });

            const data = await response.json();

            if (data.success) {
                alert(`✅ Withdrawal successful!\n\nReceived: ${received.toFixed(2)} KZD\nTransaction: ${receipt.hash.substring(0, 10)}...`);
                setAmount('');
                fetchBalance();
            } else {
                alert('Failed to record withdrawal: ' + data.error);
            }

        } catch (error) {
            console.error('Withdrawal error:', error);
            alert('Withdrawal failed: ' + (error.message || 'Unknown error'));
        } finally {
            setIsWithdrawing(false);
        }
    };

    if (!account) {
        return (
            <>
                <Head>
                    <title>Withdraw - KUZADESIGN</title>
                </Head>
                <div className="container">
                    <div className="page-header">
                        <h1 className="page-title">💰 Withdraw</h1>
                    </div>
                    <div className="card mt-3 text-center">
                        <p>Please connect your wallet to withdraw</p>
                        <button className="btn btn-primary mt-2" onClick={() => window.location.href = '/'}>
                            Connect Wallet
                        </button>
                    </div>
                </div>
                <BottomNav />
            </>
        );
    }

    return (
        <>
            <Head>
                <title>Withdraw - KUZADESIGN</title>
                <meta name="viewport" content="width=device-width, initial-scale=1.0, maximum-scale=1.0, user-scalable=no" />
            </Head>

            <div className="container">
                <div className="page-header">
                    <h1 className="page-title">💰 Withdraw</h1>
                    <p className="page-subtitle">Withdraw your earnings</p>
                </div>

                {/* Tabs */}
                <div className="card fade-in">
                    <div style={{ display: 'grid', gridTemplateColumns: '1fr 1fr 1fr', gap: 8, marginBottom: 16 }}>
                        <button
                            className={`btn ${activeTab === 'daily' ? 'btn-primary' : 'btn-outline'}`}
                            style={{ padding: '10px 8px', fontSize: 12 }}
                            onClick={() => setActiveTab('daily')}
                        >
                            Daily
                        </button>
                        <button
                            className={`btn ${activeTab === 'referral' ? 'btn-primary' : 'btn-outline'}`}
                            style={{ padding: '10px 8px', fontSize: 12 }}
                            onClick={() => setActiveTab('referral')}
                        >
                            Referral
                        </button>
                        <button
                            className={`btn ${activeTab === 'manager' ? 'btn-primary' : 'btn-outline'}`}
                            style={{ padding: '10px 8px', fontSize: 12 }}
                            onClick={() => setActiveTab('manager')}
                        >
                            Manager
                        </button>
                    </div>

                    {/* Available Balance */}
                    <div style={{
                        padding: 16,
                        background: 'linear-gradient(135deg, #667eea 0%, #764ba2 100%)',
                        borderRadius: 12,
                        color: 'white',
                        marginBottom: 16
                    }}>
                        <div style={{ opacity: 0.9, fontSize: 14, marginBottom: 4 }}>Available Balance</div>
                        <div style={{ fontSize: 32, fontWeight: 700 }}>{getAvailableBalance().toFixed(2)} KZD</div>
                    </div>

                    {/* Withdraw Amount */}
                    <div className="input-group">
                        <label className="input-label">Withdrawal Amount (minimum 10 KZD)</label>
                        <input
                            type="number"
                            className="input"
                            placeholder="Enter amount"
                            value={amount}
                            onChange={(e) => setAmount(e.target.value)}
                            min="10"
                            step="0.01"
                        />
                    </div>

                    {/* Fee Preview */}
                    {amount && parseFloat(amount) >= 10 && (
                        <div style={{
                            padding: 12,
                            background: 'var(--bg-secondary)',
                            borderRadius: 8,
                            marginBottom: 16
                        }}>
                            <div className="flex flex-between mb-1">
                                <span className="text-sm">Withdrawal Amount</span>
                                <span className="text-bold">{parseFloat(amount).toFixed(2)} KZD</span>
                            </div>
                            <div className="flex flex-between mb-1">
                                <span className="text-sm">Fee (5%)</span>
                                <span className="text-bold" style={{ color: 'var(--error)' }}>
                                    -{(parseFloat(amount) * 0.05).toFixed(2)} KZD
                                </span>
                            </div>
                            <div style={{ borderTop: '1px solid var(--primary-100)', marginTop: 8, paddingTop: 8 }}>
                                <div className="flex flex-between">
                                    <span className="text-bold">You will receive</span>
                                    <span className="text-bold" style={{ color: 'var(--success)', fontSize: 18 }}>
                                        {(parseFloat(amount) * 0.95).toFixed(2)} KZD
                                    </span>
                                </div>
                            </div>
                        </div>
                    )}

                    {/* Quick Amount Buttons */}
                    <div className="grid grid-2 mb-2" style={{ gap: 8 }}>
                        <button
                            className="btn btn-outline"
                            style={{ fontSize: 12, padding: '8px' }}
                            onClick={() => setAmount('10')}
                        >
                            Min (10 KZD)
                        </button>
                        <button
                            className="btn btn-outline"
                            style={{ fontSize: 12, padding: '8px' }}
                            onClick={() => setAmount(getAvailableBalance().toString())}
                        >
                            Max ({getAvailableBalance().toFixed(2)} KZD)
                        </button>
                    </div>

                    {/* Withdraw Button */}
                    <button
                        className="btn btn-accent ripple"
                        style={{ width: '100%' }}
                        onClick={handleWithdraw}
                        disabled={isWithdrawing || !amount || parseFloat(amount) < 10}
                    >
                        {isWithdrawing ? (
                            <><div className="spinner" style={{ width: 20, height: 20, borderWidth: 2 }}></div> Processing...</>
                        ) : (
                            '💸 Withdraw Now'
                        )}
                    </button>
                </div>

                {/* Info */}
                <div className="card mt-2" style={{ background: '#fff3e0' }}>
                    <div style={{ fontSize: 20, marginBottom: 8 }}>ℹ️</div>
                    <div className="text-bold mb-1">Important Notes:</div>
                    <ul style={{ paddingLeft: 20, margin: 0, fontSize: 14 }}>
                        <li>Minimum withdrawal: 10 KZD</li>
                        <li>Admin fee: 5% (deducted from balance)</li>
                        <li>Withdrawal processed on blockchain</li>
                        <li>Transaction may take 1-2 minutes</li>
                    </ul>
                </div>
            </div>

            <BottomNav />
        </>
    );
}
