import { useState, useEffect } from 'react';
import Head from 'next/head';
import WalletConnect from '../components/WalletConnect';
import BottomNav from '../components/BottomNav';

export default function Team() {
    const [account, setAccount] = useState(null);
    const [stats, setStats] = useState(null);
    const [loading, setLoading] = useState(false);
    const [copied, setCopied] = useState(false);

    useEffect(() => {
        if (account) {
            fetchStats();
        }
    }, [account]);

    const fetchStats = async () => {
        setLoading(true);
        try {
            const response = await fetch(`/api/team/stats?userWallet=${account}`);
            const data = await response.json();
            if (data.success) {
                setStats(data);
            } else {
                console.error('Failed to fetch stats:', data);
            }
        } catch (error) {
            console.error('Error fetching stats:', error);
        } finally {
            setLoading(false);
        }
    };

    const copyReferralLink = () => {
        const link = `${window.location.origin}/?ref=${stats.user.referralCode}`;
        navigator.clipboard.writeText(link);
        setCopied(true);
        setTimeout(() => setCopied(false), 2000);
    };

    return (
        <>
            <Head>
                <title>Team - KUZADESIGN</title>
                <meta name="viewport" content="width=device-width, initial-scale=1.0, maximum-scale=1.0, user-scalable=no" />
            </Head>

            <div className="container">
                <div className="page-header">
                    <h1 className="page-title">👥 My Team</h1>
                    <p className="page-subtitle">Referral & Manager Stats</p>
                </div>

                <WalletConnect onConnect={setAccount} />

                {!account && (
                    <div className="card mt-3 text-center">
                        <div style={{ fontSize: 48, marginBottom: 16 }}>👥</div>
                        <h3 style={{ marginBottom: 8 }}>Connect Your Wallet</h3>
                        <p style={{ color: 'var(--text-secondary)', marginBottom: 20 }}>
                            Connect your wallet to view your referral team and manager stats
                        </p>
                    </div>
                )}

                {account && loading && (
                    <div className="loading-container">
                        <div className="spinner"></div>
                    </div>
                )}

                {account && !loading && stats && (
                    <>
                        {/* Referral Link */}
                        <div className="card fade-in mt-2">
                            <div className="card-header">🔗 Your Referral Link</div>
                            <div className="card-body">
                                <div style={{
                                    background: 'var(--bg-secondary)',
                                    padding: 12,
                                    borderRadius: 8,
                                    fontFamily: 'monospace',
                                    fontSize: 12,
                                    wordBreak: 'break-all',
                                    marginBottom: 12
                                }}>
                                    {window.location.origin}/?ref={stats.user.referralCode}
                                </div>
                                <button
                                    className="btn btn-primary ripple"
                                    style={{ width: '100%' }}
                                    onClick={copyReferralLink}
                                >
                                    {copied ? '✅ Copied!' : '📋 Copy Link'}
                                </button>
                            </div>
                        </div>

                        {/* Referral Stats */}
                        <div className="card mt-2">
                            <div className="card-header">📊 Referral Statistics</div>
                            <div className="card-body">
                                <div className="grid grid-2 mb-2" style={{ gap: 12 }}>
                                    <div style={{ padding: 12, background: 'rgba(102, 126, 234, 0.2)', borderRadius: 8 }}>
                                        <div className="text-sm text-secondary">Total Referrals</div>
                                        <div className="text-xl text-bold">{stats.referral.totalReferrals}</div>
                                    </div>
                                    <div style={{ padding: 12, background: 'rgba(0, 230, 118, 0.2)', borderRadius: 8 }}>
                                        <div className="text-sm text-secondary">Total Earnings</div>
                                        <div className="text-xl text-bold" style={{ color: 'var(--accent)' }}>
                                            {stats.referral.totalEarnings.toFixed(2)} KZD
                                        </div>
                                    </div>
                                </div>

                                <div className="text-bold mb-2">Levels Breakdown:</div>
                                {[1, 2, 3, 4, 5].map((level) => (
                                    <div key={level} className="mb-2" style={{
                                        padding: 12,
                                        background: 'var(--bg-secondary)',
                                        borderRadius: 8
                                    }}>
                                        <div className="flex flex-between">
                                            <span>Level {level} ({[8, 5, 3, 2, 1][level - 1]}%)</span>
                                            <span className="text-bold">{stats.referral.levels[level] || 0} members</span>
                                        </div>
                                    </div>
                                ))}
                            </div>
                        </div>

                        {/* Manager Stats */}
                        <div className="card mt-2">
                            <div className="card-header">👑 Manager Status</div>
                            <div className="card-body">
                                <div className="flex flex-between mb-2">
                                    <span>Current Level</span>
                                    <span className="badge badge-primary">
                                        {stats.manager.currentLevel === 0 ? 'No Level' : `Manager ${stats.manager.currentLevel}`}
                                    </span>
                                </div>

                                <div className="mb-2">
                                    <div className="flex flex-between mb-1">
                                        <span className="text-sm">Group Turnover</span>
                                        <span className="text-bold">{stats.manager.groupTurnover.toFixed(2)} KZD</span>
                                    </div>
                                    {stats.manager.nextLevel && (
                                        <>
                                            <div className="progress">
                                                <div
                                                    className="progress-bar"
                                                    style={{ width: `${stats.manager.progress}%` }}
                                                ></div>
                                            </div>
                                            <div className="text-sm text-secondary mt-1">
                                                Next level: Manager {stats.manager.nextLevel} at {stats.manager.nextLevelTurnover} KZD
                                            </div>
                                        </>
                                    )}
                                </div>

                                <div style={{
                                    padding: 12,
                                    background: 'rgba(255, 167, 38, 0.2)',
                                    borderRadius: 8,
                                    borderLeft: '4px solid var(--warning)'
                                }}>
                                    <div className="text-sm text-secondary">Total Manager Bonus</div>
                                    <div className="text-xl text-bold" style={{ color: 'var(--warning)' }}>
                                        {stats.manager.totalBonus.toFixed(2)} KZD
                                    </div>
                                </div>

                                <div className="mt-2">
                                    <div className="text-bold text-sm mb-1">Manager Levels:</div>
                                    {[
                                        { level: 1, turnover: 1000, bonus: 2 },
                                        { level: 2, turnover: 3000, bonus: 3.5 },
                                        { level: 3, turnover: 5000, bonus: 6 },
                                        { level: 4, turnover: 7500, bonus: 10 },
                                        { level: 5, turnover: 10000, bonus: 15 },
                                    ].map((m) => (
                                        <div key={m.level} className="text-sm mb-1" style={{
                                            padding: 8,
                                            background: stats.manager.claimedLevels[`level${m.level}`] ? 'rgba(0, 230, 118, 0.2)' : 'var(--bg-secondary)',
                                            borderRadius: 6,
                                            display: 'flex',
                                            justifyContent: 'space-between'
                                        }}>
                                            <span>Manager {m.level}: {m.turnover} KZD</span>
                                            <span>
                                                {stats.manager.claimedLevels[`level${m.level}`] ? '✅' : `${m.bonus}%`}
                                            </span>
                                        </div>
                                    ))}
                                </div>
                            </div>
                        </div>
                    </>
                )}
            </div>

            <BottomNav />
        </>
    );
}
