import { useState, useEffect } from 'react';
import Head from 'next/head';
import BottomNav from '../components/BottomNav';
import { PACKAGES } from '../lib/web3Config';

export default function Stake() {
    const [account, setAccount] = useState(null);
    const [selectedPackage, setSelectedPackage] = useState(null);
    const [isApproving, setIsApproving] = useState(false);
    const [isStaking, setIsStaking] = useState(false);

    useEffect(() => {
        checkWallet();
    }, []);

    const checkWallet = async () => {
        if (typeof window.ethereum !== 'undefined') {
            const accounts = await window.ethereum.request({ method: 'eth_accounts' });
            if (accounts.length > 0) {
                setAccount(accounts[0]);
            }
        }
    };

    const handleStake = async (pkg) => {
        if (!account) {
            alert('Please connect wallet first');
            window.location.href = '/';
            return;
        }

        setSelectedPackage(pkg);

        if (confirm(`Stake ${pkg.amount} KZD in ${pkg.name} package?\n\nDaily Rate: ${pkg.rate}%\nDuration: ${pkg.days} days\nTotal Return: ${(pkg.amount * pkg.rate * pkg.days / 100).toFixed(2)} KZD`)) {
            await executeStake(pkg);
        }
    };

    const executeStake = async (pkg) => {
        try {
            setIsStaking(true);

            const { ethers } = await import('ethers');
            const provider = new ethers.BrowserProvider(window.ethereum);
            const signer = await provider.getSigner();

            // Get contract addresses
            const TOKEN_ADDRESS = process.env.NEXT_PUBLIC_KZD_TOKEN_ADDRESS;
            const STAKING_ADDRESS = process.env.NEXT_PUBLIC_STAKING_CONTRACT_ADDRESS;

            console.log('Token Address:', TOKEN_ADDRESS);
            console.log('Staking Address:', STAKING_ADDRESS);

            if (!TOKEN_ADDRESS || !STAKING_ADDRESS ||
                TOKEN_ADDRESS === '0x0000000000000000000000000000000000000000') {
                alert('⚠️ Contract not configured!\n\nPlease deploy contracts first or check .env configuration.');
                setIsStaking(false);
                return;
            }

            // Import ABIs
            const { KZD_TOKEN_ABI, STAKING_CONTRACT_ABI } = await import('../lib/web3Config');

            const tokenContract = new ethers.Contract(TOKEN_ADDRESS, KZD_TOKEN_ABI, signer);
            const stakingContract = new ethers.Contract(STAKING_ADDRESS, STAKING_CONTRACT_ABI, signer);

            const amountToStake = ethers.parseEther(pkg.amount.toString());

            // Check balance
            console.log('Checking token balance...');
            const balance = await tokenContract.balanceOf(account);
            console.log('Balance:', ethers.formatEther(balance), 'KZD');

            if (balance < amountToStake) {
                alert(`❌ Insufficient Balance!\n\nRequired: ${pkg.amount} KZD\nYour Balance: ${ethers.formatEther(balance)} KZD\n\nPlease get KZD tokens first.`);
                setIsStaking(false);
                return;
            }

            // Step 1: Check current allowance
            console.log('Checking allowance...');
            const currentAllowance = await tokenContract.allowance(account, STAKING_ADDRESS);
            console.log('Current Allowance:', ethers.formatEther(currentAllowance), 'KZD');

            // Step 2: Approve if needed
            if (currentAllowance < amountToStake) {
                console.log('Approving tokens...');
                setIsApproving(true);
                const approveTx = await tokenContract.approve(STAKING_ADDRESS, amountToStake);
                console.log('Approval TX:', approveTx.hash);
                await approveTx.wait();
                console.log('Approval confirmed!');
                setIsApproving(false);
            } else {
                console.log('Already approved, skipping approval step');
            }

            // Step 3: Stake
            console.log('Staking...');
            const stakeTx = await stakingContract.stake(pkg.id);
            console.log('Stake TX:', stakeTx.hash);
            const receipt = await stakeTx.wait();
            console.log('Stake confirmed!', receipt);

            // Step 4: Record in database
            console.log('Recording to database...');
            const response = await fetch('/api/stake/create', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({
                    userWallet: account,
                    packageId: pkg.id,
                    txHash: receipt.hash,
                }),
            });

            const data = await response.json();

            if (data.success) {
                alert(`✅ Staking Success!\n\n${pkg.amount} KZD staked in ${pkg.name}\n\nTX: ${receipt.hash.substring(0, 16)}...`);
                window.location.href = '/';
            } else {
                alert('⚠️ Database Error: ' + data.error + '\n\nStake transaction successful but failed to record. Please contact support.');
            }

        } catch (error) {
            console.error('Staking error:', error);

            let errorMessage = 'Unknown error';

            if (error.message.includes('user rejected')) {
                errorMessage = 'Transaction cancelled by user';
            } else if (error.message.includes('insufficient funds')) {
                errorMessage = 'Insufficient BNB for gas fees';
            } else if (error.message.includes('execution reverted')) {
                errorMessage = 'Contract error - Please check:\n• You have enough KZD tokens\n• Contract is not paused\n• Package is valid';
            } else {
                errorMessage = error.message;
            }

            alert('❌ Staking Failed!\n\n' + errorMessage);
        } finally {
            setIsStaking(false);
            setIsApproving(false);
            setSelectedPackage(null);
        }
    };

    return (
        <>
            <Head>
                <title>Stake - KUZADESIGN</title>
                <meta name="viewport" content="width=device-width, initial-scale=1.0, maximum-scale=1.0, user-scalable=no" />
            </Head>

            <div className="container">
                <div className="page-header">
                    <h1 className="page-title">💎 Staking Packages</h1>
                    <p className="page-subtitle">Choose your staking plan</p>
                </div>

                {PACKAGES.map((pkg) => (
                    <div key={pkg.id} className="card fade-in mt-2" style={{ borderLeft: `4px solid ${pkg.color}` }}>
                        <div className="flex flex-between mb-2">
                            <h3 style={{ margin: 0, color: pkg.color }}>{pkg.name}</h3>
                            <span className="badge badge-primary">{pkg.rate}% Daily</span>
                        </div>

                        <div className="grid grid-2 mb-2" style={{ gap: 12 }}>
                            <div>
                                <div className="text-sm text-secondary">Stake Amount</div>
                                <div className="text-lg text-bold">{pkg.amount} KZD</div>
                            </div>
                            <div>
                                <div className="text-sm text-secondary">Duration</div>
                                <div className="text-lg text-bold">{pkg.days} Days</div>
                            </div>
                        </div>

                        <div className="mb-2" style={{ padding: 12, background: 'var(--bg-secondary)', borderRadius: 8 }}>
                            <div className="text-sm text-secondary mb-1">Total Return</div>
                            <div className="text-xl text-bold" style={{ color: pkg.color }}>
                                {(pkg.amount * pkg.rate * pkg.days / 100).toFixed(2)} KZD
                            </div>
                            <div className="text-sm text-secondary">
                                Profit: +{((pkg.amount * pkg.rate * pkg.days / 100) - pkg.amount).toFixed(2)} KZD ({((pkg.rate * pkg.days) - 100).toFixed(0)}%)
                            </div>
                        </div>

                        <button
                            className="btn btn-primary ripple"
                            style={{ width: '100%', background: `linear-gradient(135deg, ${pkg.color} 0%, ${pkg.color}dd 100%)` }}
                            onClick={() => handleStake(pkg)}
                            disabled={isStaking && selectedPackage?.id === pkg.id}
                        >
                            {isStaking && selectedPackage?.id === pkg.id ? (
                                isApproving ? (
                                    <><div className="spinner" style={{ width: 20, height: 20, borderWidth: 2 }}></div> Approving...</>
                                ) : (
                                    <><div className="spinner" style={{ width: 20, height: 20, borderWidth: 2 }}></div> Staking...</>
                                )
                            ) : (
                                `Stake ${pkg.amount} KZD`
                            )}
                        </button>
                    </div>
                ))}

                <div className="card mt-2" style={{ background: 'var(--primary-50)' }}>
                    <div style={{ fontSize: 24, marginBottom: 8 }}>ℹ️</div>
                    <div className="text-bold mb-1">How it works:</div>
                    <ul style={{ paddingLeft: 20, margin: 0, color: 'var(--text-secondary)' }}>
                        <li>Choose a staking package</li>
                        <li>Approve KZD token spending</li>
                        <li>Confirm staking transaction</li>
                        <li>Earn daily rewards automatically</li>
                        <li>Withdraw anytime (minimum 10 KZD)</li>
                    </ul>
                </div>
            </div>

            <BottomNav />
        </>
    );
}
