import { useState, useEffect } from 'react';
import Head from 'next/head';
import BottomNav from '../components/BottomNav';

export default function Profile() {
    const [account, setAccount] = useState(null);
    const [stakes, setStakes] = useState([]);
    const [transactions, setTransactions] = useState([]);
    const [loading, setLoading] = useState(false);

    useEffect(() => {
        checkWallet();
    }, []);

    useEffect(() => {
        if (account) {
            fetchData();
        }
    }, [account]);

    const checkWallet = async () => {
        if (typeof window.ethereum !== 'undefined') {
            const accounts = await window.ethereum.request({ method: 'eth_accounts' });
            if (accounts.length > 0) {
                setAccount(accounts[0]);
            }
        }
    };

    const fetchData = async () => {
        setLoading(true);
        try {
            // Fetch stakes
            const stakesRes = await fetch(`/api/stake/list?userWallet=${account}`);
            const stakesData = await stakesRes.json();
            if (stakesData.success) {
                setStakes(stakesData.stakes);
            }

            // Fetch transactions
            const txRes = await fetch(`/api/transactions/history?userWallet=${account}&limit=20`);
            const txData = await txRes.json();
            if (txData.success) {
                setTransactions(txData.transactions);
            }
        } catch (error) {
            console.error('Error fetching data:', error);
        } finally {
            setLoading(false);
        }
    };

    const getTypeLabel = (type) => {
        const labels = {
            stake: 'Stake',
            withdraw_daily: 'Withdraw Daily',
            withdraw_referral: 'Withdraw Referral',
            withdraw_manager: 'Withdraw Manager',
            referral_bonus: 'Referral Bonus',
            manager_bonus: 'Manager Bonus',
        };
        return labels[type] || type;
    };

    const getTypeColor = (type) => {
        if (type === 'stake') return '#2196f3';
        if (type.startsWith('withdraw')) return '#f44336';
        return '#4caf50';
    };

    if (!account) {
        return (
            <>
                <Head>
                    <title>Profile - KUZADESIGN</title>
                </Head>
                <div className="container">
                    <div className="page-header">
                        <h1 className="page-title">👤 Profile</h1>
                    </div>
                    <div className="card mt-3 text-center">
                        <p>Please connect your wallet to view profile</p>
                        <button className="btn btn-primary mt-2" onClick={() => window.location.href = '/'}>
                            Connect Wallet
                        </button>
                    </div>
                </div>
                <BottomNav />
            </>
        );
    }

    return (
        <>
            <Head>
                <title>Profile - KUZADESIGN</title>
                <meta name="viewport" content="width=device-width, initial-scale=1.0, maximum-scale=1.0, user-scalable=no" />
            </Head>

            <div className="container">
                <div className="page-header">
                    <h1 className="page-title">👤 My Profile</h1>
                    <p className="page-subtitle">Staking history & transactions</p>
                </div>

                {/* Wallet Info */}
                <div className="card fade-in">
                    <div className="text-sm text-secondary mb-1">Wallet Address</div>
                    <div style={{
                        fontFamily: 'monospace',
                        fontSize: 14,
                        fontWeight: 600,
                        wordBreak: 'break-all',
                        padding: 12,
                        background: 'var(--bg-secondary)',
                        borderRadius: 8
                    }}>
                        {account}
                    </div>
                </div>

                {/* My Stakes */}
                <div className="card mt-2">
                    <div className="card-header">📦 My Stakes ({stakes.length})</div>
                    <div className="card-body">
                        {loading ? (
                            <div className="text-center"><div className="spinner"></div></div>
                        ) : stakes.length === 0 ? (
                            <div className="text-center text-secondary">No stakes yet</div>
                        ) : (
                            stakes.map((stake, idx) => (
                                <div key={idx} className="mb-2" style={{
                                    padding: 12,
                                    background: 'var(--bg-secondary)',
                                    borderRadius: 8,
                                    borderLeft: `4px solid ${stake.status === 'active' ? '#4caf50' : '#757575'}`
                                }}>
                                    <div className="flex flex-between mb-1">
                                        <span className="text-bold">{stake.packageName}</span>
                                        <span className={`badge ${stake.status === 'active' ? 'badge-success' : 'badge-secondary'}`}>
                                            {stake.status}
                                        </span>
                                    </div>
                                    <div className="text-sm mb-1">
                                        Amount: <span className="text-bold">{stake.amount} KZD</span> •
                                        Rate: <span className="text-bold">{stake.dailyRate / 100}%</span>
                                    </div>
                                    {stake.status === 'active' && (
                                        <>
                                            <div className="progress">
                                                <div
                                                    className="progress-bar"
                                                    style={{ width: `${((stake.durationDays - stake.daysRemaining) / stake.durationDays * 100)}%` }}
                                                ></div>
                                            </div>
                                            <div className="text-sm text-secondary mt-1">
                                                {stake.daysRemaining} days remaining
                                            </div>
                                        </>
                                    )}
                                    <div className="text-sm mt-1">
                                        Claimed: <span className="text-bold" style={{ color: '#4caf50' }}>
                                            {stake.totalClaimed.toFixed(2)} KZD
                                        </span>
                                    </div>
                                </div>
                            ))
                        )}
                    </div>
                </div>

                {/* Transaction History */}
                <div className="card mt-2">
                    <div className="card-header">📜 Recent Transactions</div>
                    <div className="card-body">
                        {loading ? (
                            <div className="text-center"><div className="spinner"></div></div>
                        ) : transactions.length === 0 ? (
                            <div className="text-center text-secondary">No transactions yet</div>
                        ) : (
                            transactions.map((tx, idx) => (
                                <div key={idx} className="mb-2" style={{
                                    padding: 12,
                                    background: 'var(--bg-secondary)',
                                    borderRadius: 8,
                                    borderLeft: `4px solid ${getTypeColor(tx.type)}`
                                }}>
                                    <div className="flex flex-between mb-1">
                                        <span className="text-bold" style={{ fontSize: 13 }}>
                                            {getTypeLabel(tx.type)}
                                        </span>
                                        <span className={`badge badge-${tx.status === 'success' ? 'success' : 'warning'}`}>
                                            {tx.status}
                                        </span>
                                    </div>
                                    <div className="text-sm">
                                        Amount: <span className="text-bold">{tx.amount.toFixed(2)} KZD</span>
                                        {tx.fee > 0 && (
                                            <> • Fee: <span style={{ color: 'var(--error)' }}>{tx.fee.toFixed(2)} KZD</span></>
                                        )}
                                    </div>
                                    {tx.txHash && (
                                        <div className="text-sm mt-1" style={{
                                            fontFamily: 'monospace',
                                            color: 'var(--primary-600)',
                                            cursor: 'pointer'
                                        }} onClick={() => window.open(`https://testnet.bscscan.com/tx/${tx.txHash}`, '_blank')}>
                                            {tx.txHash.substring(0, 10)}...{tx.txHash.substring(tx.txHash.length - 8)} ↗
                                        </div>
                                    )}
                                    <div className="text-sm text-secondary mt-1">
                                        {new Date(tx.createdAt).toLocaleDateString('id-ID', {
                                            day: '2-digit',
                                            month: 'short',
                                            year: 'numeric',
                                            hour: '2-digit',
                                            minute: '2-digit'
                                        })}
                                    </div>
                                </div>
                            ))
                        )}
                    </div>
                </div>
            </div>

            <BottomNav />
        </>
    );
}
