import { useState, useEffect } from 'react';
import Head from 'next/head';
import WalletConnect from '../components/WalletConnect';
import BottomNav from '../components/BottomNav';

export default function Home() {
    const [account, setAccount] = useState(null);
    const [balance, setBalance] = useState({ dailyRewards: 0, referralBonus: 0, managerBonus: 0, total: 0 });
    const [stakes, setStakes] = useState([]);
    const [loading, setLoading] = useState(false);

    useEffect(() => {
        if (account) {
            fetchData();
        }
    }, [account]);

    const fetchData = async () => {
        setLoading(true);
        try {
            // Fetch balance
            const balRes = await fetch(`/api/balance/get?userWallet=${account}`);
            const balData = await balRes.json();
            if (balData.success) {
                setBalance(balData.balance);
            }

            // Fetch stakes
            const stakesRes = await fetch(`/api/stake/list?userWallet=${account}`);
            const stakesData = await stakesRes.json();
            if (stakesData.success) {
                setStakes(stakesData.stakes.filter(s => s.status === 'active'));
            }
        } catch (error) {
            console.error('Error fetching data:', error);
        } finally {
            setLoading(false);
        }
    };

    return (
        <>
            <Head>
                <title>KUZADESIGN Staking</title>
                <meta name="viewport" content="width=device-width, initial-scale=1.0, maximum-scale=1.0, user-scalable=no" />
            </Head>

            <div className="container">
                <div className="page-header">
                    <h1 className="page-title">KUZADESIGN</h1>
                    <p className="page-subtitle">Web3 Staking Platform</p>
                </div>

                <WalletConnect onConnect={setAccount} />

                {account && (
                    <div className="fade-in">
                        {loading ? (
                            <div className="loading-container">
                                <div className="spinner"></div>
                            </div>
                        ) : (
                            <>
                                {/* Balance Cards */}
                                <div className="grid grid-2 mt-3">
                                    <div className="card" style={{ background: 'linear-gradient(135deg, #667eea 0%, #764ba2 100%)', color: 'white' }}>
                                        <div className="text-sm" style={{ opacity: 0.9 }}>Total Balance</div>
                                        <div className="text-xl text-bold mt-1">{balance.total.toFixed(2)} KZD</div>
                                    </div>
                                    <div className="card" style={{ background: 'linear-gradient(135deg, #f093fb 0%, #f5576c 100%)', color: 'white' }}>
                                        <div className="text-sm" style={{ opacity: 0.9 }}>Active Stakes</div>
                                        <div className="text-xl text-bold mt-1">{stakes.length}</div>
                                    </div>
                                </div>

                                {/* Balance Details */}
                                <div className="card mt-2">
                                    <div className="card-header">💰 Withdrawable Balance</div>
                                    <div className="card-body">
                                        <div className="flex flex-between mb-2">
                                            <span>Daily Rewards</span>
                                            <span className="text-bold">{balance.dailyRewards.toFixed(2)} KZD</span>
                                        </div>
                                        <div className="flex flex-between mb-2">
                                            <span>Referral Bonus</span>
                                            <span className="text-bold">{balance.referralBonus.toFixed(2)} KZD</span>
                                        </div>
                                        <div className="flex flex-between">
                                            <span>Manager Bonus</span>
                                            <span className="text-bold">{balance.managerBonus.toFixed(2)} KZD</span>
                                        </div>
                                    </div>
                                </div>

                                {/* Active Stakes */}
                                {stakes.length > 0 && (
                                    <div className="card mt-2">
                                        <div className="card-header">📊 Active Stakes</div>
                                        <div className="card-body">
                                            {stakes.slice(0, 3).map((stake, idx) => (
                                                <div key={idx} className="mb-2" style={{ paddingBottom: 12, borderBottom: idx < 2 ? '1px solid var(--primary-100)' : 'none' }}>
                                                    <div className="flex flex-between mb-1">
                                                        <span className="badge badge-primary">{stake.packageName}</span>
                                                        <span className="text-bold">{stake.amount} KZD</span>
                                                    </div>
                                                    <div className="progress">
                                                        <div
                                                            className="progress-bar"
                                                            style={{ width: `${((stake.durationDays - stake.daysRemaining) / stake.durationDays * 100)}%` }}
                                                        ></div>
                                                    </div>
                                                    <div className="text-sm text-secondary mt-1">
                                                        {stake.daysRemaining} days remaining • {stake.dailyRate / 100}% daily
                                                    </div>
                                                </div>
                                            ))}
                                        </div>
                                    </div>
                                )}

                                {/* Quick Actions */}
                                <div className="grid grid-2 mt-2" style={{ gap: 12 }}>
                                    <button className="btn btn-primary" onClick={() => window.location.href = '/stake'}>
                                        💎 Stake Now
                                    </button>
                                    <button className="btn btn-accent" onClick={() => window.location.href = '/withdraw'}>
                                        💰 Withdraw
                                    </button>
                                </div>
                            </>
                        )}
                    </div>
                )}

                {!account && (
                    <div className="card mt-3 text-center">
                        <div style={{ fontSize: 48, marginBottom: 16 }}>👛</div>
                        <h3 style={{ marginBottom: 8 }}>Welcome to KUZADESIGN</h3>
                        <p style={{ color: 'var(--text-secondary)', marginBottom: 20 }}>
                            Connect your wallet to start staking and earn daily rewards
                        </p>
                        <div className="grid grid-2" style={{ gap: 12, fontSize: 14 }}>
                            <div>
                                <div style={{ fontSize: 24, marginBottom: 8 }}>💎</div>
                                <div className="text-bold">5 Packages</div>
                                <div className="text-sm text-secondary">1.5% - 2.5% daily</div>
                            </div>
                            <div>
                                <div style={{ fontSize: 24, marginBottom: 8 }}>🎁</div>
                                <div className="text-bold">5 Level Referral</div>
                                <div className="text-sm text-secondary">Up to 8% bonus</div>
                            </div>
                        </div>
                    </div>
                )}
            </div>

            <BottomNav />
        </>
    );
}
