import connectDB from '../../../lib/mongodb';
import Balance from '../../../models/Balance';
import Transaction from '../../../models/Transaction';

const WITHDRAWAL_FEE_PERCENT = 5;
const MIN_WITHDRAWAL = 10;

export default async function handler(req, res) {
    if (req.method !== 'POST') {
        return res.status(405).json({ error: 'Method not allowed' });
    }

    try {
        await connectDB();

        const { userWallet, amount, type, txHash } = req.body;

        // Validate inputs
        if (!userWallet || !amount || !type || !txHash) {
            return res.status(400).json({ error: 'Missing required fields' });
        }

        if (amount < MIN_WITHDRAWAL) {
            return res.status(400).json({ error: `Minimum withdrawal is ${MIN_WITHDRAWAL} KZD` });
        }

        // Validate type
        const validTypes = ['withdraw_daily', 'withdraw_referral', 'withdraw_manager'];
        if (!validTypes.includes(type)) {
            return res.status(400).json({ error: 'Invalid withdrawal type' });
        }

        // Get user balance
        const balance = await Balance.findOne({ userWallet: userWallet.toLowerCase() });

        if (!balance) {
            return res.status(400).json({ error: 'No balance found' });
        }

        // Check balance based on type
        let balanceField;
        if (type === 'withdraw_daily') balanceField = 'dailyRewards';
        else if (type === 'withdraw_referral') balanceField = 'referralBonus';
        else balanceField = 'managerBonus';

        if (balance[balanceField] < amount) {
            return res.status(400).json({ error: 'Insufficient balance' });
        }

        // Calculate fee
        const fee = (amount * WITHDRAWAL_FEE_PERCENT) / 100;
        const received = amount - fee;

        // Deduct from balance
        balance[balanceField] -= amount;
        await balance.save();

        // Create transaction record
        const transaction = new Transaction({
            userWallet: userWallet.toLowerCase(),
            type,
            amount,
            fee,
            received,
            txHash,
            status: 'success',
        });

        await transaction.save();

        return res.status(200).json({
            success: true,
            message: 'Withdrawal processed successfully',
            withdrawal: {
                amount,
                fee,
                received,
                txHash,
                remainingBalance: balance[balanceField],
            },
        });

    } catch (error) {
        console.error('Withdrawal error:', error);
        return res.status(500).json({ error: 'Internal server error' });
    }
}
