import connectDB from '../../../lib/mongodb';
import { getReferralTree } from '../../../lib/referralSystem';
import { getManagerStats } from '../../../lib/managerSystem';
import User from '../../../models/User';

export default async function handler(req, res) {
    if (req.method !== 'GET') {
        return res.status(405).json({ error: 'Method not allowed' });
    }

    try {
        await connectDB();

        const { userWallet } = req.query;

        if (!userWallet) {
            return res.status(400).json({ error: 'Wallet address required' });
        }

        // Get user or create if not exists
        let user = await User.findOne({ walletAddress: userWallet.toLowerCase() });

        if (!user) {
            // Auto-create user if not exists
            user = await User.create({
                walletAddress: userWallet.toLowerCase(),
                referralCode: userWallet.toLowerCase(), // Use full wallet address
                referrerAddress: null,
            });
            console.log('Created new user:', userWallet);
        }

        // Handle existing users who might have old short codes
        // We will just use their wallet address for display regardless
        const displayReferralCode = user.walletAddress;

        // Get referral tree
        const referralTree = await getReferralTree(userWallet);

        // Get manager stats
        const managerStats = await getManagerStats(userWallet);

        return res.status(200).json({
            success: true,
            user: {
                walletAddress: user.walletAddress,
                referralCode: user.walletAddress, // Always return wallet address as code
            },
            referral: referralTree,
            manager: managerStats,
        });

    } catch (error) {
        console.error('Team stats error:', error);
        return res.status(500).json({
            success: false,
            error: 'Internal server error',
            message: error.message
        });
    }
}
