import connectDB from '../../../lib/mongodb';
import User from '../../../models/User';
import Stake from '../../../models/Stake';
import { distributeReferralBonus } from '../../../lib/referralSystem';
import { updateGroupTurnover } from '../../../lib/managerSystem';

const PACKAGES = {
    0: { name: 'BASIC', amount: 10, dailyRate: 150, durationDays: 90 },
    1: { name: 'STANDARD', amount: 50, dailyRate: 160, durationDays: 80 },
    2: { name: 'PRO', amount: 100, dailyRate: 160, durationDays: 70 },
    3: { name: 'MINI PRO', amount: 500, dailyRate: 190, durationDays: 60 },
    4: { name: 'MAX PRO', amount: 1000, dailyRate: 250, durationDays: 50 },
};

export default async function handler(req, res) {
    if (req.method !== 'POST') {
        return res.status(405).json({ error: 'Method not allowed' });
    }

    try {
        await connectDB();

        const { userWallet, packageId, txHash } = req.body;

        if (!userWallet || packageId === undefined || !txHash) {
            return res.status(400).json({ error: 'Missing required fields' });
        }

        // Validate package
        const pkg = PACKAGES[packageId];
        if (!pkg) {
            return res.status(400).json({ error: 'Invalid package ID' });
        }

        // Check if stake with this txHash already exists
        const existingStake = await Stake.findOne({ txHash });
        if (existingStake) {
            return res.status(400).json({ error: 'Stake already recorded' });
        }

        // Create stake record
        const startDate = new Date();
        const endDate = new Date(startDate.getTime() + pkg.durationDays * 24 * 60 * 60 * 1000);

        const stake = new Stake({
            userWallet: userWallet.toLowerCase(),
            packageId,
            packageName: pkg.name,
            amount: pkg.amount,
            dailyRate: pkg.dailyRate,
            durationDays: pkg.durationDays,
            startDate,
            endDate,
            txHash,
            status: 'active',
        });

        await stake.save();

        // Distribute referral bonuses (5 levels)
        await distributeReferralBonus(userWallet, pkg.amount);

        // Update group turnover and check manager bonuses
        await updateGroupTurnover(userWallet, pkg.amount);

        return res.status(201).json({
            success: true,
            message: 'Stake created successfully',
            stake: {
                packageName: stake.packageName,
                amount: stake.amount,
                startDate: stake.startDate,
                endDate: stake.endDate,
                dailyRate: stake.dailyRate,
            },
        });

    } catch (error) {
        console.error('Stake creation error:', error);
        return res.status(500).json({ error: 'Internal server error' });
    }
}
