import connectDB from '../../../../lib/mongodb';
import { isOwner } from '../../../../lib/adminAuth';
import User from '../../../../models/User';
import Stake from '../../../../models/Stake';
import Transaction from '../../../../models/Transaction';

export default async function handler(req, res) {
    if (req.method !== 'GET') {
        return res.status(405).json({ error: 'Method not allowed' });
    }

    try {
        await connectDB();

        const { userWallet, page = 1, limit = 10, search = '' } = req.query;

        if (!userWallet) {
            return res.status(400).json({ error: 'Wallet address required' });
        }

        // Check owner
        if (!isOwner(userWallet)) {
            return res.status(403).json({ error: 'Unauthorized' });
        }

        // Build query
        const query = {};
        if (search) {
            query.$or = [
                { walletAddress: { $regex: search, $options: 'i' } },
                { referralCode: { $regex: search, $options: 'i' } }
            ];
        }

        // Pagination
        const skip = (page - 1) * limit;

        // Fetch users
        const users = await User.find(query)
            .sort({ createdAt: -1 })
            .skip(skip)
            .limit(parseInt(limit));

        const totalUsers = await User.countDocuments(query);

        // Enhance user data with stats
        const enhancedUsers = await Promise.all(users.map(async (user) => {
            const activeStakes = await Stake.countDocuments({
                userWallet: user.walletAddress,
                status: 'active'
            });

            const totalStaked = await Stake.aggregate([
                { $match: { userWallet: user.walletAddress } },
                { $group: { _id: null, total: { $sum: '$amount' } } }
            ]);

            const earnings = await Transaction.aggregate([
                {
                    $match: {
                        userWallet: user.walletAddress,
                        type: { $in: ['daily_reward', 'referral_bonus', 'manager_bonus'] },
                        status: 'success'
                    }
                },
                { $group: { _id: null, total: { $sum: '$amount' } } }
            ]);

            return {
                ...user.toObject(),
                activeStakes,
                totalStaked: totalStaked[0]?.total || 0,
                totalEarnings: earnings[0]?.total || 0
            };
        }));

        return res.status(200).json({
            success: true,
            users: enhancedUsers,
            pagination: {
                total: totalUsers,
                page: parseInt(page),
                pages: Math.ceil(totalUsers / limit)
            }
        });

    } catch (error) {
        console.error('Admin user list error:', error);
        return res.status(500).json({ error: 'Internal server error' });
    }
}
