import connectDB from '../../../lib/mongodb';
import { isOwner } from '../../../lib/adminAuth';
import User from '../../../models/User';
import Stake from '../../../models/Stake';
import Balance from '../../../models/Balance';
import Transaction from '../../../models/Transaction';

export default async function handler(req, res) {
    if (req.method !== 'GET') {
        return res.status(405).json({ error: 'Method not allowed' });
    }

    try {
        await connectDB();

        const { userWallet } = req.query;

        if (!userWallet) {
            return res.status(400).json({ error: 'Wallet address required' });
        }

        // Check if owner
        if (!isOwner(userWallet)) {
            return res.status(403).json({
                success: false,
                error: 'Unauthorized - Owner access only'
            });
        }

        // Get total users
        const totalUsers = await User.countDocuments();

        // Get total stakes
        const allStakes = await Stake.find();
        const activeStakes = allStakes.filter(s => s.status === 'active');
        const completedStakes = allStakes.filter(s => s.status === 'completed');

        const totalStakedAmount = allStakes.reduce((sum, stake) => sum + stake.amount, 0);
        const activeStakedAmount = activeStakes.reduce((sum, stake) => sum + stake.amount, 0);

        // Get total rewards distributed
        const transactions = await Transaction.find({
            type: { $in: ['referral_bonus', 'manager_bonus'] },
            status: 'success'
        });
        const totalRewardsDistributed = transactions.reduce((sum, tx) => sum + tx.amount, 0);

        // Get total balances
        const balances = await Balance.find();
        const totalDailyRewards = balances.reduce((sum, b) => sum + (b.dailyRewards || 0), 0);
        const totalReferralBonus = balances.reduce((sum, b) => sum + (b.referralBonus || 0), 0);
        const totalManagerBonus = balances.reduce((sum, b) => sum + (b.managerBonus || 0), 0);

        // Get pending withdrawals
        const pendingWithdrawals = await Transaction.find({
            type: { $in: ['withdraw_daily', 'withdraw_referral', 'withdraw_manager'] },
            status: 'pending'
        });

        const stats = {
            users: {
                total: totalUsers,
                withStakes: activeStakes.length > 0 ? new Set(activeStakes.map(s => s.userWallet)).size : 0
            },
            stakes: {
                total: allStakes.length,
                active: activeStakes.length,
                completed: completedStakes.length,
                totalAmount: totalStakedAmount,
                activeAmount: activeStakedAmount
            },
            rewards: {
                totalDistributed: totalRewardsDistributed,
                daily: totalDailyRewards,
                referral: totalReferralBonus,
                manager: totalManagerBonus,
                total: totalDailyRewards + totalReferralBonus + totalManagerBonus
            },
            withdrawals: {
                pending: pendingWithdrawals.length,
                pendingAmount: pendingWithdrawals.reduce((sum, tx) => sum + tx.amount, 0)
            }
        };

        return res.status(200).json({
            success: true,
            stats
        });

    } catch (error) {
        console.error('Admin stats error:', error);
        return res.status(500).json({
            success: false,
            error: 'Internal server error',
            message: error.message
        });
    }
}
