import { useState, useEffect } from 'react';
import Head from 'next/head';
import { useRouter } from 'next/router';
import WalletConnect from '../../components/WalletConnect';
import BottomNav from '../../components/BottomNav';

export default function AdminUsers() {
    const router = useRouter();
    const [account, setAccount] = useState(null);
    const [users, setUsers] = useState([]);
    const [pagination, setPagination] = useState(null);
    const [loading, setLoading] = useState(false);
    const [searchTerm, setSearchTerm] = useState('');
    const [debouncedSearch, setDebouncedSearch] = useState('');

    // Debounce search
    useEffect(() => {
        const timer = setTimeout(() => {
            setDebouncedSearch(searchTerm);
        }, 500);
        return () => clearTimeout(timer);
    }, [searchTerm]);

    useEffect(() => {
        if (account) {
            fetchUsers(1, debouncedSearch);
        }
    }, [account, debouncedSearch]);

    const fetchUsers = async (page = 1, search = '') => {
        setLoading(true);
        try {
            const query = new URLSearchParams({
                userWallet: account,
                page,
                limit: 10, // Show 10 users per page
                search
            });

            const response = await fetch(`/api/admin/users/list?${query}`);
            const data = await response.json();

            if (data.success) {
                setUsers(data.users);
                setPagination(data.pagination);
            } else {
                if (data.error === 'Unauthorized') {
                    router.push('/');
                }
            }
        } catch (error) {
            console.error('Error fetching users:', error);
        } finally {
            setLoading(false);
        }
    };

    return (
        <>
            <Head>
                <title>User Management - Admin</title>
                <meta name="viewport" content="width=device-width, initial-scale=1.0, maximum-scale=1.0, user-scalable=no" />
            </Head>

            <div className="container">
                <div className="page-header">
                    <button className="back-btn" onClick={() => router.push('/admin')}>
                        ← Back
                    </button>
                    <h1 className="page-title">👥 User Management</h1>
                </div>

                <WalletConnect onConnect={setAccount} />

                {account && (
                    <>
                        {/* Search Bar */}
                        <div className="card mt-2">
                            <input
                                type="text"
                                className="form-input"
                                placeholder="🔍 Search wallet or referral code..."
                                value={searchTerm}
                                onChange={(e) => setSearchTerm(e.target.value)}
                                style={{ width: '100%' }}
                            />
                        </div>

                        {/* User List */}
                        {loading ? (
                            <div className="loading-container">
                                <div className="spinner"></div>
                            </div>
                        ) : (
                            <div className="mt-2">
                                <div style={{ color: 'var(--text-secondary)', marginBottom: 8, fontSize: 14 }}>
                                    Total Users: {pagination?.total || 0}
                                </div>

                                {users.length === 0 ? (
                                    <div className="card text-center text-secondary">No users found</div>
                                ) : (
                                    users.map((user) => (
                                        <div key={user._id} className="card mb-2 fade-in">
                                            <div className="flex flex-between mb-1">
                                                <span className="badge badge-primary">
                                                    {user.referralCode}
                                                </span>
                                                <span className="text-sm text-secondary">
                                                    {new Date(user.createdAt).toLocaleDateString()}
                                                </span>
                                            </div>

                                            <div className="text-sm monospace mb-2" style={{ wordBreak: 'break-all' }}>
                                                {user.walletAddress}
                                            </div>

                                            <div className="grid grid-3" style={{ gap: 8, borderTop: '1px solid var(--border-color)', paddingTop: 8 }}>
                                                <div>
                                                    <div className="text-xs text-secondary">Staked</div>
                                                    <div className="text-bold">{user.totalStaked.toFixed(0)}</div>
                                                </div>
                                                <div>
                                                    <div className="text-xs text-secondary">Active</div>
                                                    <div className="text-bold" style={{ color: 'var(--success)' }}>
                                                        {user.activeStakes}
                                                    </div>
                                                </div>
                                                <div>
                                                    <div className="text-xs text-secondary">Earned</div>
                                                    <div className="text-bold" style={{ color: 'var(--warning)' }}>
                                                        {user.totalEarnings.toFixed(2)}
                                                    </div>
                                                </div>
                                            </div>

                                            {user.managerLevel > 0 && (
                                                <div className="mt-2 text-xs" style={{
                                                    background: 'rgba(255, 167, 38, 0.1)',
                                                    padding: 4,
                                                    borderRadius: 4,
                                                    display: 'inline-block',
                                                    color: 'var(--warning)'
                                                }}>
                                                    👑 Manager Level {user.managerLevel}
                                                </div>
                                            )}
                                        </div>
                                    ))
                                )}

                                {/* Pagination */}
                                {pagination && pagination.pages > 1 && (
                                    <div className="flex flex-center mt-3 mb-4" style={{ gap: 10 }}>
                                        <button
                                            className="btn btn-sm"
                                            disabled={pagination.page === 1}
                                            onClick={() => fetchUsers(pagination.page - 1, debouncedSearch)}
                                        >
                                            Previous
                                        </button>
                                        <span className="text-sm">
                                            Page {pagination.page} of {pagination.pages}
                                        </span>
                                        <button
                                            className="btn btn-sm"
                                            disabled={pagination.page === pagination.pages}
                                            onClick={() => fetchUsers(pagination.page + 1, debouncedSearch)}
                                        >
                                            Next
                                        </button>
                                    </div>
                                )}
                            </div>
                        )}
                    </>
                )}
            </div>

            <BottomNav />
        </>
    );
}
