import { useState, useEffect } from 'react';
import Head from 'next/head';
import { useRouter } from 'next/router';
import WalletConnect from '../../components/WalletConnect';
import BottomNav from '../../components/BottomNav';

export default function AdminSettings() {
    const router = useRouter();
    const [account, setAccount] = useState(null);
    const [loading, setLoading] = useState(false);
    const [saving, setSaving] = useState(false);

    // Initial State structure matches MongoDB model
    const [settings, setSettings] = useState({
        referralLevels: { 1: 5, 2: 3, 3: 1, 4: 0.5, 5: 0.5 },
        managerLevels: {
            1: { turnover: 10000, bonus: 2 },
            2: { turnover: 25000, bonus: 1.5 },
            3: { turnover: 50000, bonus: 1 },
            4: { turnover: 100000, bonus: 0.5 },
            5: { turnover: 250000, bonus: 0.5 }
        },
        staking: {
            minWithdrawal: 10,
            withdrawalFee: 5
        }
    });

    useEffect(() => {
        if (account) {
            fetchSettings();
        }
    }, [account]);

    const fetchSettings = async () => {
        setLoading(true);
        try {
            const response = await fetch(`/api/admin/settings?userWallet=${account}`);
            const data = await response.json();

            if (data.success) {
                // Merge active settings
                setSettings(prev => ({
                    ...prev,
                    referralLevels: { ...prev.referralLevels, ...data.settings.referralLevels },
                    managerLevels: { ...prev.managerLevels, ...data.settings.managerLevels },
                    staking: { ...prev.staking, ...data.settings.staking }
                }));
            } else if (data.error === 'Unauthorized - Owner access only') {
                router.push('/');
            }
        } catch (error) {
            console.error('Error fetching settings:', error);
        } finally {
            setLoading(false);
        }
    };

    const handleSave = async () => {
        setSaving(true);
        try {
            const response = await fetch('/api/admin/settings', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({
                    userWallet: account,
                    ...settings
                })
            });

            const data = await response.json();
            if (data.success) {
                alert('✅ Settings updated successfully!');
            } else {
                alert('❌ Error: ' + data.error);
            }
        } catch (error) {
            console.error('Error saving settings:', error);
            alert('Failed to save settings');
        } finally {
            setSaving(false);
        }
    };

    const handleChange = (section, key, subKey, value) => {
        setSettings(prev => {
            const newSettings = { ...prev };
            if (section === 'referralLevels') {
                newSettings.referralLevels[key] = parseFloat(value);
            } else if (section === 'staking') {
                newSettings.staking[key] = parseFloat(value);
            } else if (section === 'managerLevels') {
                newSettings.managerLevels[key] = {
                    ...newSettings.managerLevels[key],
                    [subKey]: parseFloat(value)
                };
            }
            return newSettings;
        });
    };

    return (
        <>
            <Head>
                <title>Settings - Admin</title>
                <meta name="viewport" content="width=device-width, initial-scale=1.0, maximum-scale=1.0, user-scalable=no" />
            </Head>

            <div className="container">
                <div className="page-header">
                    <button className="back-btn" onClick={() => router.push('/admin')}>
                        ← Back
                    </button>
                    <h1 className="page-title">⚙️ System Settings</h1>
                </div>

                <WalletConnect onConnect={setAccount} />

                {account && (
                    <>
                        {loading ? (
                            <div className="loading-container">
                                <div className="spinner"></div>
                            </div>
                        ) : (
                            <div className="animate-fade-in">
                                {/* Referral Levels */}
                                <div className="card mb-2">
                                    <h3 className="mb-2">📣 Referral Commission (%)</h3>
                                    <div className="grid grid-2" style={{ gap: 10 }}>
                                        {[1, 2, 3, 4, 5].map(level => (
                                            <div key={level}>
                                                <label className="text-sm text-secondary">Level {level}</label>
                                                <input
                                                    type="number"
                                                    className="form-input"
                                                    value={settings.referralLevels[level]}
                                                    onChange={(e) => handleChange('referralLevels', level, null, e.target.value)}
                                                />
                                            </div>
                                        ))}
                                    </div>
                                </div>

                                {/* Manager Levels */}
                                <div className="card mb-2">
                                    <h3 className="mb-2">👑 Manager Levels</h3>
                                    {[1, 2, 3, 4, 5].map(level => (
                                        <div key={level} className="mb-2" style={{ padding: 10, background: 'var(--bg-secondary)', borderRadius: 8 }}>
                                            <div className="text-bold mb-1">Level {level}</div>
                                            <div className="grid grid-2" style={{ gap: 10 }}>
                                                <div>
                                                    <label className="text-xs text-secondary">Turnover (KZD)</label>
                                                    <input
                                                        type="number"
                                                        className="form-input"
                                                        value={settings.managerLevels[level].turnover}
                                                        onChange={(e) => handleChange('managerLevels', level, 'turnover', e.target.value)}
                                                    />
                                                </div>
                                                <div>
                                                    <label className="text-xs text-secondary">Bonus (%)</label>
                                                    <input
                                                        type="number"
                                                        className="form-input"
                                                        value={settings.managerLevels[level].bonus}
                                                        onChange={(e) => handleChange('managerLevels', level, 'bonus', e.target.value)}
                                                    />
                                                </div>
                                            </div>
                                        </div>
                                    ))}
                                </div>

                                {/* Staking Settings */}
                                <div className="card mb-3">
                                    <h3 className="mb-2">💰 Withdrawal Settings</h3>
                                    <div className="grid grid-2" style={{ gap: 10 }}>
                                        <div>
                                            <label className="text-sm text-secondary">Min Withdraw (KZD)</label>
                                            <input
                                                type="number"
                                                className="form-input"
                                                value={settings.staking.minWithdrawal}
                                                onChange={(e) => handleChange('staking', 'minWithdrawal', null, e.target.value)}
                                            />
                                        </div>
                                        <div>
                                            <label className="text-sm text-secondary">Fee (%)</label>
                                            <input
                                                type="number"
                                                className="form-input"
                                                value={settings.staking.withdrawalFee}
                                                onChange={(e) => handleChange('staking', 'withdrawalFee', null, e.target.value)}
                                            />
                                        </div>
                                    </div>
                                </div>

                                {/* Fund Contract Section */}
                                <div className="card mb-3" style={{ border: '1px solid var(--primary)' }}>
                                    <h3 className="mb-2">🏦 Fund Smart Contract</h3>
                                    <p className="text-sm text-secondary mb-2">
                                        Top up contract balance for user withdrawals.
                                    </p>
                                    <div className="flex" style={{ gap: 10 }}>
                                        <input
                                            type="number"
                                            className="form-input"
                                            placeholder="Amount (KZD)"
                                            id="fundAmount"
                                            style={{ flex: 1 }}
                                        />
                                        <button
                                            className="btn btn-primary"
                                            onClick={async () => {
                                                const amount = document.getElementById('fundAmount').value;
                                                if (!amount) return alert('Please enter amount');

                                                try {
                                                    setSaving(true);
                                                    const { ethers } = await import('ethers');
                                                    const provider = new ethers.BrowserProvider(window.ethereum);
                                                    const signer = await provider.getSigner();

                                                    // Get contracts
                                                    const TOKEN_ADDRESS = process.env.NEXT_PUBLIC_KZD_TOKEN_ADDRESS;
                                                    const STAKING_ADDRESS = process.env.NEXT_PUBLIC_STAKING_CONTRACT_ADDRESS;

                                                    const { KZD_TOKEN_ABI, STAKING_CONTRACT_ABI } = await import('../../lib/web3Config');

                                                    const tokenContract = new ethers.Contract(TOKEN_ADDRESS, KZD_TOKEN_ABI, signer);
                                                    const stakingContract = new ethers.Contract(STAKING_ADDRESS, STAKING_CONTRACT_ABI, signer);

                                                    const weiAmount = ethers.parseEther(amount);

                                                    // Approve
                                                    alert('Please confirm Approval transaction...');
                                                    const approveTx = await tokenContract.approve(STAKING_ADDRESS, weiAmount);
                                                    await approveTx.wait();

                                                    // Fund
                                                    alert('Please confirm Fund transaction...');
                                                    const fundTx = await stakingContract.fundContract(weiAmount);
                                                    await fundTx.wait();

                                                    alert('✅ Contract funded successfully!');
                                                    document.getElementById('fundAmount').value = '';
                                                } catch (error) {
                                                    console.error(error);
                                                    alert('Error: ' + (error.reason || error.message));
                                                } finally {
                                                    setSaving(false);
                                                }
                                            }}
                                            disabled={saving}
                                        >
                                            💰 Fund
                                        </button>
                                    </div>
                                </div>

                                {/* Staking Packages (Smart Contract) */}
                                <div className="card mb-3">
                                    <h3 className="mb-2">💎 Staking Packages (On-Chain)</h3>
                                    <p className="text-sm text-secondary mb-2">
                                        Update package parameters directly on Smart Contract.
                                    </p>

                                    <div className="text-xs text-warning mb-2 p-2" style={{ background: 'rgba(255,167,38,0.1)', borderRadius: 4 }}>
                                        ⚠️ Warning: Changing packages affects NEW stakes only. Existing stakes remain unchanged.
                                    </div>

                                    {[0, 1, 2, 3, 4].map((pkgId) => (
                                        <div key={pkgId} className="mb-2 p-2" style={{ background: 'var(--bg-secondary)', borderRadius: 8 }}>
                                            <div className="flex flex-between mb-2">
                                                <div className="text-bold">Package ID: {pkgId}</div>
                                                <button
                                                    className="btn btn-sm btn-outline"
                                                    onClick={async () => {
                                                        const name = prompt(`Update Name for ID ${pkgId}`);
                                                        const amount = prompt(`Update Amount (KZD) for ID ${pkgId}`);
                                                        const rate = prompt(`Update Daily Rate (bps, 150 = 1.5%) for ID ${pkgId}`);
                                                        const days = prompt(`Update Duration (Days) for ID ${pkgId}`);

                                                        if (!name || !amount || !rate || !days) return;

                                                        try {
                                                            setSaving(true);
                                                            const { ethers } = await import('ethers');
                                                            const provider = new ethers.BrowserProvider(window.ethereum);
                                                            const signer = await provider.getSigner();
                                                            const STAKING_ADDRESS = process.env.NEXT_PUBLIC_STAKING_CONTRACT_ADDRESS;
                                                            const { STAKING_CONTRACT_ABI } = await import('../../lib/web3Config');
                                                            const stakingContract = new ethers.Contract(STAKING_ADDRESS, STAKING_CONTRACT_ABI, signer);

                                                            const tx = await stakingContract.updatePackage(
                                                                pkgId,
                                                                name,
                                                                ethers.parseEther(amount),
                                                                parseInt(rate),
                                                                parseInt(days * 24 * 3600), // Convert days to seconds
                                                                true // isActive
                                                            );
                                                            await tx.wait();
                                                            alert('✅ Package updated!');
                                                        } catch (error) {
                                                            console.error(error);
                                                            alert('Error: ' + error.message);
                                                        } finally {
                                                            setSaving(false);
                                                        }
                                                    }}
                                                >
                                                    ✏️ Edit
                                                </button>
                                            </div>
                                            <div className="text-sm text-secondary">
                                                Use "Edit" button to update Name, Amount, Rate, and Duration on blockchain.
                                            </div>
                                        </div>
                                    ))}
                                </div>

                                {/* Save Button */}
                                <button
                                    className="btn btn-primary ripple mb-4"
                                    style={{ width: '100%', padding: 15 }}
                                    onClick={handleSave}
                                    disabled={saving}
                                >
                                    {saving ? (
                                        <><div className="spinner" style={{ width: 20, height: 20 }}></div> Processing...</>
                                    ) : (
                                        '💾 Save Database Settings'
                                    )}
                                </button>
                            </div>
                        )}
                    </>
                )}
            </div>

            <BottomNav />
        </>
    );
}
