import { useState, useEffect } from 'react';
import Head from 'next/head';
import { useRouter } from 'next/router';
import WalletConnect from '../../components/WalletConnect';
import BottomNav from '../../components/BottomNav';

export default function AdminDashboard() {
    const router = useRouter();
    const [account, setAccount] = useState(null);
    const [stats, setStats] = useState(null);
    const [loading, setLoading] = useState(false);
    const [isOwner, setIsOwner] = useState(false);

    useEffect(() => {
        if (account) {
            checkOwnerAndFetchStats();
        }
    }, [account]);

    const checkOwnerAndFetchStats = async () => {
        setLoading(true);
        try {
            const response = await fetch(`/api/admin/stats?userWallet=${account}`);
            const data = await response.json();

            if (data.success) {
                setIsOwner(true);
                setStats(data.stats);
            } else {
                // Not owner, redirect to home
                setIsOwner(false);
                setTimeout(() => router.push('/'), 2000);
            }
        } catch (error) {
            console.error('Error fetching admin stats:', error);
            setIsOwner(false);
        } finally {
            setLoading(false);
        }
    };

    return (
        <>
            <Head>
                <title>Admin Dashboard - KUZADESIGN</title>
                <meta name="viewport" content="width=device-width, initial-scale=1.0, maximum-scale=1.0, user-scalable=no" />
            </Head>

            <div className="container">
                <div className="page-header">
                    <h1 className="page-title">⚙️ Admin Dashboard</h1>
                    <p className="page-subtitle">System Management</p>
                </div>

                <WalletConnect onConnect={setAccount} />

                {!account && (
                    <div className="card mt-3 text-center">
                        <div style={{ fontSize: 48, marginBottom: 16 }}>🔐</div>
                        <h3 style={{ marginBottom: 8 }}>Owner Access Required</h3>
                        <p style={{ color: 'var(--text-secondary)', marginBottom: 20 }}>
                            Connect the owner wallet to access admin dashboard
                        </p>
                    </div>
                )}

                {account && !isOwner && !loading && (
                    <div className="card mt-3 text-center" style={{
                        background: 'rgba(239, 83, 80, 0.1)',
                        borderLeft: '4px solid var(--error)'
                    }}>
                        <div style={{ fontSize: 48, marginBottom: 16 }}>❌</div>
                        <h3 style={{ marginBottom: 8, color: 'var(--error)' }}>Unauthorized</h3>
                        <p style={{ color: 'var(--text-secondary)' }}>
                            This wallet is not authorized for admin access. Redirecting...
                        </p>
                    </div>
                )}

                {account && loading && (
                    <div className="loading-container">
                        <div className="spinner"></div>
                    </div>
                )}

                {account && isOwner && stats && !loading && (
                    <>
                        {/* Admin Badge */}
                        <div className="card mt-2" style={{
                            background: 'var(--gradient-primary)',
                            color: 'white'
                        }}>
                            <div className="flex flex-between">
                                <div>
                                    <div className="text-sm" style={{ opacity: 0.9 }}>Admin Mode Active</div>
                                    <div className="text-lg text-bold mt-1">Owner Dashboard</div>
                                </div>
                                <div style={{ fontSize: 32 }}>👑</div>
                            </div>
                        </div>

                        {/* Quick Stats */}
                        <div className="grid grid-2 mt-2" style={{ gap: 12 }}>
                            <div className="card" style={{ background: 'rgba(102, 126, 234, 0.2)' }}>
                                <div className="text-sm text-secondary">Total Users</div>
                                <div className="text-xl text-bold">{stats.users.total}</div>
                                <div className="text-sm text-secondary">{stats.users.withStakes} with stakes</div>
                            </div>
                            <div className="card" style={{ background: 'rgba(0, 230, 118, 0.2)' }}>
                                <div className="text-sm text-secondary">Active Stakes</div>
                                <div className="text-xl text-bold">{stats.stakes.active}</div>
                                <div className="text-sm text-secondary">{stats.stakes.completed} completed</div>
                            </div>
                        </div>

                        {/* Staking Stats */}
                        <div className="card mt-2">
                            <div className="card-header">💎 Staking Statistics</div>
                            <div className="card-body">
                                <div className="flex flex-between mb-2">
                                    <span>Total Staked</span>
                                    <span className="text-bold">{stats.stakes.totalAmount.toFixed(2)} KZD</span>
                                </div>
                                <div className="flex flex-between mb-2">
                                    <span>Currently Active</span>
                                    <span className="text-bold" style={{ color: 'var(--accent)' }}>
                                        {stats.stakes.activeAmount.toFixed(2)} KZD
                                    </span>
                                </div>
                                <div className="flex flex-between">
                                    <span>Avg per Stake</span>
                                    <span className="text-bold">
                                        {stats.stakes.total > 0 ? (stats.stakes.totalAmount / stats.stakes.total).toFixed(2) : '0.00'} KZD
                                    </span>
                                </div>
                            </div>
                        </div>

                        {/* Rewards Stats */}
                        <div className="card mt-2">
                            <div className="card-header">🎁 Rewards Distribution</div>
                            <div className="card-body">
                                <div className="mb-2" style={{
                                    padding: 12,
                                    background: 'var(--gradient-success)',
                                    borderRadius: 8,
                                    color: 'white'
                                }}>
                                    <div className="text-sm" style={{ opacity: 0.9 }}>Total Distributed</div>
                                    <div className="text-xl text-bold">{stats.rewards.total.toFixed(2)} KZD</div>
                                </div>
                                <div className="flex flex-between mb-1">
                                    <span className="text-sm">Daily Rewards</span>
                                    <span className="text-sm text-bold">{stats.rewards.daily.toFixed(2)} KZD</span>
                                </div>
                                <div className="flex flex-between mb-1">
                                    <span className="text-sm">Referral Bonus</span>
                                    <span className="text-sm text-bold">{stats.rewards.referral.toFixed(2)} KZD</span>
                                </div>
                                <div className="flex flex-between">
                                    <span className="text-sm">Manager Bonus</span>
                                    <span className="text-sm text-bold">{stats.rewards.manager.toFixed(2)} KZD</span>
                                </div>
                            </div>
                        </div>

                        {/* Withdrawals */}
                        <div className="card mt-2">
                            <div className="card-header">💰 Withdrawals</div>
                            <div className="card-body">
                                <div className="flex flex-between mb-2">
                                    <span>Pending Requests</span>
                                    <span className="badge badge-warning">{stats.withdrawals.pending}</span>
                                </div>
                                <div className="flex flex-between">
                                    <span>Pending Amount</span>
                                    <span className="text-bold" style={{ color: 'var(--warning)' }}>
                                        {stats.withdrawals.pendingAmount.toFixed(2)} KZD
                                    </span>
                                </div>
                            </div>
                        </div>

                        {/* Quick Actions */}
                        <div className="grid grid-2 mt-2" style={{ gap: 12 }}>
                            <button
                                className="btn btn-primary"
                                onClick={() => router.push('/admin/settings')}
                            >
                                ⚙️ Settings
                            </button>
                            <button
                                className="btn btn-accent"
                                onClick={() => router.push('/admin/users')}
                            >
                                👥 Manage Users
                            </button>
                        </div>
                    </>
                )}
            </div>

            <BottomNav />
        </>
    );
}
