const User = require('../models/User');
const Balance = require('../models/Balance');
const Transaction = require('../models/Transaction');

// Referral bonus percentages for each level
const REFERRAL_LEVELS = {
    1: 0.08, // 8%
    2: 0.05, // 5%
    3: 0.03, // 3%
    4: 0.02, // 2%
    5: 0.01, // 1%
};

/**
 * Calculate and distribute referral bonuses (5 levels)
 * @param {string} userWallet - Wallet address of the user who staked
 * @param {number} stakeAmount - Amount staked
 * @returns {Promise<Array>} - Array of bonus distributions
 */
async function distributeReferralBonus(userWallet, stakeAmount) {
    const distributions = [];

    try {
        const user = await User.findOne({ walletAddress: userWallet.toLowerCase() });

        if (!user || !user.referrerAddress) {
            console.log('No referrer found for user:', userWallet);
            return distributions;
        }

        let currentReferrer = user.referrerAddress;

        // Loop through 5 levels
        for (let level = 1; level <= 5; level++) {
            if (!currentReferrer) break;

            const referrer = await User.findOne({ walletAddress: currentReferrer.toLowerCase() });

            if (!referrer) break;

            // Calculate bonus for this level
            const bonusAmount = stakeAmount * REFERRAL_LEVELS[level];

            // Update referrer's balance
            let balance = await Balance.findOne({ userWallet: referrer.walletAddress });

            if (!balance) {
                balance = new Balance({ userWallet: referrer.walletAddress });
            }

            balance.referralBonus += bonusAmount;
            await balance.save();

            // Log transaction
            await Transaction.create({
                userWallet: referrer.walletAddress,
                type: 'referral_bonus',
                amount: bonusAmount,
                status: 'success',
                metadata: {
                    fromUser: userWallet,
                    level: level,
                    stakeAmount: stakeAmount,
                },
            });

            distributions.push({
                referrer: referrer.walletAddress,
                level: level,
                bonus: bonusAmount,
            });

            // Move to next level upline
            currentReferrer = referrer.referrerAddress;
        }

        console.log(`✅ Distributed referral bonuses for ${userWallet}:`, distributions);
        return distributions;

    } catch (error) {
        console.error('Error distributing referral bonus:', error);
        throw error;
    }
}

/**
 * Get referral tree for a user
 * @param {string} userWallet - Wallet address
 * @param {number} maxLevel - Maximum level to fetch (default 5)
 * @returns {Promise<Object>} - Referral tree data
 */
async function getReferralTree(userWallet, maxLevel = 5) {
    try {
        const tree = {
            levels: {},
            totalReferrals: 0,
            totalEarnings: 0,
        };

        // Initialize all levels to 0
        for (let level = 1; level <= maxLevel; level++) {
            tree.levels[level] = 0;
        }

        // Get direct referrals only (level 1)
        // Note: For deeper levels, would need recursive logic
        const referrals = await User.find({
            referrerAddress: userWallet.toLowerCase(),
        });

        tree.levels[1] = referrals.length;
        tree.totalReferrals = referrals.length;

        // Get total referral earnings
        const transactions = await Transaction.find({
            userWallet: userWallet.toLowerCase(),
            type: 'referral_bonus',
            status: 'success',
        });

        tree.totalEarnings = transactions.reduce((sum, tx) => sum + tx.amount, 0);

        return tree;

    } catch (error) {
        console.error('Error getting referral tree:', error);
        // Return default structure instead of throwing
        return {
            levels: { 1: 0, 2: 0, 3: 0, 4: 0, 5: 0 },
            totalReferrals: 0,
            totalEarnings: 0,
        };
    }
}

module.exports = {
    distributeReferralBonus,
    getReferralTree,
    REFERRAL_LEVELS,
};
