const User = require('../models/User');
const Balance = require('../models/Balance');
const Transaction = require('../models/Transaction');
const Stake = require('../models/Stake');

// Manager level requirements and bonuses
const MANAGER_LEVELS = {
    1: { turnover: 1000, bonus: 0.02 },    // 1000 KZD → 2%
    2: { turnover: 3000, bonus: 0.035 },   // 3000 KZD → 3.5%
    3: { turnover: 5000, bonus: 0.06 },    // 5000 KZD → 6%
    4: { turnover: 7500, bonus: 0.10 },    // 7500 KZD → 10%
    5: { turnover: 10000, bonus: 0.15 },   // 10000 KZD → 15%
};

/**
 * Update group turnover for a user's upline chain
 * @param {string} userWallet - Wallet address of user who staked
 * @param {number} stakeAmount - Amount staked
 */
async function updateGroupTurnover(userWallet, stakeAmount) {
    try {
        const user = await User.findOne({ walletAddress: userWallet.toLowerCase() });

        if (!user || !user.referrerAddress) {
            console.log('No referrer to update turnover');
            return;
        }

        let currentReferrer = user.referrerAddress;

        // Update turnover for all uplines (unlimited levels)
        while (currentReferrer) {
            const referrer = await User.findOne({ walletAddress: currentReferrer.toLowerCase() });

            if (!referrer) break;

            // Update group turnover
            referrer.groupTurnover += stakeAmount;
            await referrer.save();

            console.log(`Updated turnover for ${referrer.walletAddress}: ${referrer.groupTurnover}`);

            // Check and award manager bonus if qualified
            await checkAndAwardManagerBonus(referrer);

            // Move to next upline
            currentReferrer = referrer.referrerAddress;
        }

    } catch (error) {
        console.error('Error updating group turnover:', error);
        throw error;
    }
}

/**
 * Check if user qualifies for manager bonus and award it (ONE-TIME per level)
 * @param {Object} user - User document from MongoDB
 */
async function checkAndAwardManagerBonus(user) {
    try {
        const currentTurnover = user.groupTurnover;
        let newManagerLevel = 0;

        // Determine highest manager level achieved
        for (let level = 5; level >= 1; level--) {
            if (currentTurnover >= MANAGER_LEVELS[level].turnover) {
                newManagerLevel = level;
                break;
            }
        }

        // If no level achieved, return
        if (newManagerLevel === 0) return;

        // Check if this manager level bonus has already been claimed
        const levelKey = `level${newManagerLevel}`;

        if (user.managerBonusClaimed.get(levelKey)) {
            console.log(`Manager level ${newManagerLevel} bonus already claimed for ${user.walletAddress}`);
            return;
        }

        // Calculate and award bonus (only if not claimed before)
        const bonusPercentage = MANAGER_LEVELS[newManagerLevel].bonus;
        const bonusAmount = MANAGER_LEVELS[newManagerLevel].turnover * bonusPercentage;

        // Update balance
        let balance = await Balance.findOne({ userWallet: user.walletAddress });

        if (!balance) {
            balance = new Balance({ userWallet: user.walletAddress });
        }

        balance.managerBonus += bonusAmount;
        await balance.save();

        // Mark this level as claimed
        user.managerBonusClaimed.set(levelKey, true);
        user.managerLevel = newManagerLevel;
        await user.save();

        // Log transaction
        await Transaction.create({
            userWallet: user.walletAddress,
            type: 'manager_bonus',
            amount: bonusAmount,
            status: 'success',
            metadata: {
                managerLevel: newManagerLevel,
                turnoverRequired: MANAGER_LEVELS[newManagerLevel].turnover,
                bonusPercentage: bonusPercentage * 100,
            },
        });

        console.log(`✅ Awarded Manager Level ${newManagerLevel} bonus to ${user.walletAddress}: ${bonusAmount} KZD`);

    } catch (error) {
        console.error('Error checking manager bonus:', error);
        throw error;
    }
}

/**
 * Get manager stats for a user
 * @param {string} userWallet - Wallet address
 * @returns {Promise<Object>} - Manager stats
 */
async function getManagerStats(userWallet) {
    try {
        const user = await User.findOne({ walletAddress: userWallet.toLowerCase() });

        if (!user) {
            return {
                currentLevel: 0,
                groupTurnover: 0,
                nextLevel: 1,
                nextLevelTurnover: MANAGER_LEVELS[1].turnover,
                progress: 0,
                totalBonus: 0,
            };
        }

        // Calculate total manager bonus earned
        const transactions = await Transaction.find({
            userWallet: userWallet.toLowerCase(),
            type: 'manager_bonus',
            status: 'success',
        });

        const totalBonus = transactions.reduce((sum, tx) => sum + tx.amount, 0);

        // Determine next level
        let nextLevel = null;
        for (let level = 1; level <= 5; level++) {
            if (user.groupTurnover < MANAGER_LEVELS[level].turnover) {
                nextLevel = level;
                break;
            }
        }

        const progress = nextLevel
            ? (user.groupTurnover / MANAGER_LEVELS[nextLevel].turnover) * 100
            : 100;

        return {
            currentLevel: user.managerLevel,
            groupTurnover: user.groupTurnover,
            nextLevel: nextLevel,
            nextLevelTurnover: nextLevel ? MANAGER_LEVELS[nextLevel].turnover : null,
            progress: Math.min(progress, 100),
            totalBonus: totalBonus,
            claimedLevels: Object.fromEntries(user.managerBonusClaimed),
        };

    } catch (error) {
        console.error('Error getting manager stats:', error);
        // Return default structure instead of throwing
        return {
            currentLevel: 0,
            groupTurnover: 0,
            nextLevel: 1,
            nextLevelTurnover: MANAGER_LEVELS[1].turnover,
            progress: 0,
            totalBonus: 0,
            claimedLevels: {},
        };
    }
}

module.exports = {
    updateGroupTurnover,
    checkAndAwardManagerBonus,
    getManagerStats,
    MANAGER_LEVELS,
};
