#!/bin/bash

# KUZADESIGN Automated Installer Script
# Run this on your Ubuntu VPS (fresh install recommended)

echo "🚀 Starting Automatic Installation..."

# 1. Update System
echo "📦 Updating Server..."
sudo apt update && sudo apt upgrade -y
sudo apt install curl git build-essential nginx certbot python3-certbot-nginx -y

# 2. Install Node.js v18
echo "🟢 Installing Node.js..."
curl -fsSL https://deb.nodesource.com/setup_18.x | sudo -E bash -
sudo apt install -y nodejs

# 3. Install MongoDB
echo "🍃 Installing MongoDB..."
curl -fsSL https://pgp.mongodb.com/server-7.0.asc | sudo gpg -o /usr/share/keyrings/mongodb-server-7.0.gpg --dearmor
echo "deb [ arch=amd64,arm64 signed-by=/usr/share/keyrings/mongodb-server-7.0.gpg ] https://repo.mongodb.org/apt/ubuntu jammy/mongodb-org/7.0 multiverse" | sudo tee /etc/apt/sources.list.d/mongodb-org-7.0.list
sudo apt update
sudo apt install -y mongodb-org
sudo systemctl start mongod
sudo systemctl enable mongod

# 4. Install PM2
echo "⚙️ Installing Process Manager (PM2)..."
sudo npm install -g pm2

# 5. Setup Project Directory
echo "📂 Setting up Project Directory..."
PROJECT_DIR="/var/www/kuzastaking"
sudo mkdir -p $PROJECT_DIR
sudo chown -R $USER:$USER $PROJECT_DIR

echo "⚠️  NOTE: Please upload your project files to $PROJECT_DIR now (Using FileZilla/SCP)"
echo "   Upload everything EXCEPT: node_modules, .git"
echo "   Waiting 10 seconds... (If files are already there, ignore this)"
sleep 10

# 6. Install Dependencies & Start App
if [ -d "$PROJECT_DIR/pages" ]; then
    echo "🚀 Installing App Dependencies..."
    cd $PROJECT_DIR
    npm install --production

    echo "▶️ Starting Application..."
    pm2 start npm --name "kuza-app" -- start
    pm2 start cron-runner.js --name "cron-kuza"
    pm2 save
    pm2 startup
else
    echo "❌ Project files not found in $PROJECT_DIR. Please upload them and run 'npm install --production' manually inside the folder."
fi

# 7. Setup Nginx Proxy
echo "🌐 Setting up Nginx..."
cat > /etc/nginx/sites-available/kuzastaking <<EOF
server {
    server_name web3-stake1.rezocrypto.us;

    location / {
        proxy_pass http://localhost:3000;
        proxy_http_version 1.1;
        proxy_set_header Upgrade \$http_upgrade;
        proxy_set_header Connection 'upgrade';
        proxy_set_header Host \$host;
        proxy_cache_bypass \$http_upgrade;
    }
}
EOF

sudo ln -s /etc/nginx/sites-available/kuzastaking /etc/nginx/sites-enabled/
sudo rm /etc/nginx/sites-enabled/default
sudo nginx -t
sudo systemctl restart nginx

# 8. SSL Setup (Interactive)
echo "🔒 Preparing SSL Setup..."
echo "After this script finishes, please run: sudo certbot --nginx -d web3-stake1.rezocrypto.us"

echo "✅ INSTALLATION COMPLETE!"
echo "Please verify your files are uploaded to $PROJECT_DIR"
