// SPDX-License-Identifier: MIT
pragma solidity ^0.8.20;

import "@openzeppelin/contracts/token/ERC20/IERC20.sol";
import "@openzeppelin/contracts/utils/ReentrancyGuard.sol";
import "@openzeppelin/contracts/access/Ownable.sol";
import "@openzeppelin/contracts/utils/Pausable.sol";

/**
 * @title KZD Staking Contract
 * @dev Contract untuk staking token KZD dengan 5 paket berbeda
 */
contract KZDStaking is ReentrancyGuard, Ownable, Pausable {
    IERC20 public kzdToken;

    // Struktur untuk paket staking
    struct Package {
        string name;
        uint256 amount;
        uint256 dailyRate; // dalam basis point (1.5% = 150, 2.5% = 250)
        uint256 durationDays;
        bool isActive;
    }

    // Struktur untuk stake individual user
    struct Stake {
        uint256 packageId;
        uint256 amount;
        uint256 startTime;
        uint256 endTime;
        bool isActive;
    }

    // Package definitions (0-4)
    mapping(uint256 => Package) public packages;
    
    // User stakes: wallet => stake array
    mapping(address => Stake[]) public userStakes;
    
    // Total staked per user
    mapping(address => uint256) public totalStaked;
    
    // Total staked in contract
    uint256 public totalStakedInContract;

    // Fee percentage (5%)
    uint256 public constant WITHDRAWAL_FEE_PERCENT = 5;
    uint256 public constant PERCENT_DENOMINATOR = 100;

    // Events
    event Staked(address indexed user, uint256 packageId, uint256 amount, uint256 startTime, uint256 endTime);
    event Withdrawn(address indexed user, uint256 amount, uint256 fee, uint256 received);
    event PackageUpdated(uint256 packageId, string name, uint256 amount, uint256 dailyRate, uint256 durationDays);

    constructor(address _kzdToken) Ownable(msg.sender) {
        kzdToken = IERC20(_kzdToken);
        
        // Initialize packages
        packages[0] = Package("BASIC", 10 * 10**18, 150, 90 days, true);        // 10 KZD, 1.5% daily, 90 days
        packages[1] = Package("STANDARD", 50 * 10**18, 160, 80 days, true);     // 50 KZD, 1.6% daily, 80 days
        packages[2] = Package("PRO", 100 * 10**18, 160, 70 days, true);         // 100 KZD, 1.6% daily, 70 days
        packages[3] = Package("MINI PRO", 500 * 10**18, 190, 60 days, true);    // 500 KZD, 1.9% daily, 60 days
        packages[4] = Package("MAX PRO", 1000 * 10**18, 250, 50 days, true);    // 1000 KZD, 2.5% daily, 50 days
    }

    /**
     * @dev Stake tokens dengan package tertentu
     * @param packageId ID package (0-4)
     */
    function stake(uint256 packageId) external nonReentrant whenNotPaused {
        require(packageId < 5, "Invalid package ID");
        Package memory pkg = packages[packageId];
        require(pkg.isActive, "Package not active");

        // Transfer token dari user ke contract
        require(kzdToken.transferFrom(msg.sender, address(this), pkg.amount), "Transfer failed");

        // Buat stake baru
        Stake memory newStake = Stake({
            packageId: packageId,
            amount: pkg.amount,
            startTime: block.timestamp,
            endTime: block.timestamp + pkg.durationDays,
            isActive: true
        });

        userStakes[msg.sender].push(newStake);
        totalStaked[msg.sender] += pkg.amount;
        totalStakedInContract += pkg.amount;

        emit Staked(msg.sender, packageId, pkg.amount, newStake.startTime, newStake.endTime);
    }

    /**
     * @dev Withdraw dengan fee 5% (dipotong dari balance contract)
     * @param amount Jumlah yang ingin di-withdraw
     */
    function withdraw(uint256 amount) external nonReentrant whenNotPaused {
        require(amount >= 10 * 10**18, "Minimum withdrawal is 10 KZD");
        
        // Hitung fee 5%
        uint256 fee = (amount * WITHDRAWAL_FEE_PERCENT) / PERCENT_DENOMINATOR;
        uint256 amountAfterFee = amount - fee;

        // Pastikan contract punya cukup balance
        require(kzdToken.balanceOf(address(this)) >= amount, "Insufficient contract balance");

        // Transfer token ke user (setelah fee)
        require(kzdToken.transfer(msg.sender, amountAfterFee), "Transfer failed");

        // Fee tetap di contract (tidak ditransfer kemana-mana, hanya dipotong)

        emit Withdrawn(msg.sender, amount, fee, amountAfterFee);
    }

    /**
     * @dev Get total stakes untuk user
     */
    function getUserStakesCount(address user) external view returns (uint256) {
        return userStakes[user].length;
    }

    /**
     * @dev Get detail stake berdasarkan index
     */
    function getUserStake(address user, uint256 index) external view returns (
        uint256 packageId,
        uint256 amount,
        uint256 startTime,
        uint256 endTime,
        bool isActive
    ) {
        require(index < userStakes[user].length, "Invalid stake index");
        Stake memory s = userStakes[user][index];
        return (s.packageId, s.amount, s.startTime, s.endTime, s.isActive);
    }

    /**
     * @dev Get package details
     */
    function getPackage(uint256 packageId) external view returns (
        string memory name,
        uint256 amount,
        uint256 dailyRate,
        uint256 durationDays,
        bool isActive
    ) {
        Package memory pkg = packages[packageId];
        return (pkg.name, pkg.amount, pkg.dailyRate, pkg.durationDays, pkg.isActive);
    }

    /**
     * @dev Update package (only owner)
     */
    function updatePackage(
        uint256 packageId,
        string memory name,
        uint256 amount,
        uint256 dailyRate,
        uint256 durationDays,
        bool isActive
    ) external onlyOwner {
        packages[packageId] = Package(name, amount, dailyRate, durationDays, isActive);
        emit PackageUpdated(packageId, name, amount, dailyRate, durationDays);
    }

    /**
     * @dev Emergency withdraw oleh owner (untuk fee yang terkumpul)
     */
    function emergencyWithdraw(uint256 amount) external onlyOwner {
        require(kzdToken.transfer(owner(), amount), "Transfer failed");
    }

    /**
     * @dev Fund contract dengan KZD tokens (untuk liquidity withdrawal)
     * @param amount Jumlah KZD yang akan didepositkan ke contract
     */
    function fundContract(uint256 amount) external onlyOwner {
        require(amount > 0, "Amount must be greater than 0");
        require(kzdToken.transferFrom(msg.sender, address(this), amount), "Transfer failed");
        
        emit ContractFunded(msg.sender, amount, kzdToken.balanceOf(address(this)));
    }

    // Event untuk fund contract
    event ContractFunded(address indexed funder, uint256 amount, uint256 newBalance);

    /**
     * @dev Pause/Unpause contract
     */
    function pause() external onlyOwner {
        _pause();
    }

    function unpause() external onlyOwner {
        _unpause();
    }

    /**
     * @dev Get contract token balance
     */
    function getContractBalance() external view returns (uint256) {
        return kzdToken.balanceOf(address(this));
    }
}
