import { useState, useEffect } from 'react';

export default function WalletConnect({ onConnect }) {
    const [account, setAccount] = useState(null);
    const [isConnecting, setIsConnecting] = useState(false);

    useEffect(() => {
        checkIfWalletIsConnected();
    }, []);

    const checkIfWalletIsConnected = async () => {
        if (typeof window.ethereum !== 'undefined') {
            try {
                const accounts = await window.ethereum.request({ method: 'eth_accounts' });
                if (accounts.length > 0) {
                    setAccount(accounts[0]);
                    if (onConnect) onConnect(accounts[0]);
                }
            } catch (error) {
                console.error('Error checking wallet:', error);
            }
        }
    };

    const connectWallet = async () => {
        if (typeof window.ethereum === 'undefined') {
            alert('Please install MetaMask!');
            return;
        }

        setIsConnecting(true);

        try {
            // Request accounts
            const accounts = await window.ethereum.request({ method: 'eth_requestAccounts' });

            // Check network
            const chainId = await window.ethereum.request({ method: 'eth_chainId' });

            // BSC Testnet = 0x61 (97 in decimal)
            if (chainId !== '0x61') {
                try {
                    await window.ethereum.request({
                        method: 'wallet_switchEthereumChain',
                        params: [{ chainId: '0x61' }],
                    });
                } catch (switchError) {
                    // Chain not added, so add it
                    if (switchError.code === 4902) {
                        await window.ethereum.request({
                            method: 'wallet_addEthereumChain',
                            params: [{
                                chainId: '0x61',
                                chainName: 'BSC Testnet',
                                nativeCurrency: {
                                    name: 'BNB',
                                    symbol: 'BNB',
                                    decimals: 18
                                },
                                rpcUrls: ['https://data-seed-prebsc-1-s1.binance.org:8545/'],
                                blockExplorerUrls: ['https://testnet.bscscan.com/']
                            }],
                        });
                    }
                }
            }

            setAccount(accounts[0]);
            if (onConnect) onConnect(accounts[0]);

            // Register user
            await fetch('/api/auth/register', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({ walletAddress: accounts[0] }),
            });

        } catch (error) {
            console.error('Error connecting wallet:', error);
            alert('Failed to connect wallet');
        } finally {
            setIsConnecting(false);
        }
    };

    const disconnectWallet = () => {
        setAccount(null);
        if (onConnect) onConnect(null);
    };

    return (
        <div>
            {!account ? (
                <button
                    className="btn btn-primary btn-block"
                    onClick={connectWallet}
                    disabled={isConnecting}
                >
                    {isConnecting ? (
                        <><div className="spinner" style={{ width: 20, height: 20, borderWidth: 2 }}></div> Connecting...</>
                    ) : (
                        <>🔗 Connect Wallet</>
                    )}
                </button>
            ) : (
                <div className="card" style={{ marginBottom: 16 }}>
                    <div style={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center' }}>
                        <div>
                            <div style={{ fontSize: 12, color: 'var(--text-secondary)', marginBottom: 4 }}>
                                Connected Wallet
                            </div>
                            <div style={{ fontSize: 14, fontWeight: 600, fontFamily: 'monospace' }}>
                                {account.substring(0, 6)}...{account.substring(account.length - 4)}
                            </div>
                        </div>
                        <button
                            className="btn btn-outline"
                            style={{ padding: '8px 16px', fontSize: 12 }}
                            onClick={disconnectWallet}
                        >
                            Disconnect
                        </button>
                    </div>
                </div>
            )}
        </div>
    );
}
